##START#OF#VOLUME_RENDERING##################################################
# geossav_volume_rendering.tcl
#
# Source script creating the volume_rendering widget in GEOSSAV, and running 
# the rendering of data. The called c/c++ procs are in the geossav.c file.
# Developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the VOLUME_RENDERING
#
global exedir
source $exedir/Geossav_volume_rendering_globals.tcl

set renderpar(datafl) "*.out"                     ;# simulated realizations file, simplified Geo-EAS format
set renderpar(nx) 100                             ;# number of grid cells (x axis)
set renderpar(ny) 100                             ;# number of grid cells (y axis)
set renderpar(nz) 10                              ;# number of grid cells (y axis)
set renderpar(nsim) 1                             ;# number of simulations
set renderpar(vartype) 0                          ;# variable type: 1=continuous (cdf), 0=categorical (pdf)
set renderpar(xsiz) 10                            ;# size of grid cell in x-direction
set renderpar(ysiz) 10                            ;# size of grid cell in y-direction
set renderpar(zsiz) 10                            ;# size of grid cell in z-direction
set renderpar(ncat) 3                             ;# number of thresholds or categories
set renderpar(cat) 1                              ;# threshold values or category codes
set renderpar(infofl) "*.out"                     ;# information file to the realization file, simplified Geo-EAS format
set renderpar(xy_plane) 1                         ;# selection of xy plane: 1 yes, 0 no
set renderpar(xz_plane) 0                         ;# selection of xz plane: 1 yes, 0 no
set renderpar(yz_plane) 0                         ;# selection of yz plane: 1 yes, 0 no


set params3d(xy_nn) 0
set params3d(xz_nn) 0
set params3d(yz_nn) 0
set params3d(plane) "XY"
set 3d_select(xy) 0
set 3d_select(xz) 0
set 3d_select(yz) 0
#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc oglswap { widget } {
    global params3d
    global 3d_select

    $widget activate
    set cmdstr "geossav_3dview 1"
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    eval $cmdstr

    $widget swapbuffers
}
#============================================================================


#############################################################################
proc read_volume_rendering_infofl_params { fname } {
    global renderpar
    global datafl_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	set renderpar(datafl) "*.out"
	$datafl_w configure -textvariable renderpar(datafl)
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\nNo such file or directory."
	return 0
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "information for sisim data export" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Information for SISIM data export" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File content error" \
		-icon error \
		-message "Information to the simulated realizations file can not be found.\n"
	    return 0
	}
    }

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(vartype) [format "%d" [string range $line 0 $space]]

    ### reads the nsim: number of realizations
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(nsim) [format "%d" [string range $line 0 $space]]

    ### reads the nx, xsiz: number, size of grid cells (x direction)
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(nx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set renderpar(xsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the ny, ysiz: number, size of grid cells (y direction)
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(ny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set renderpar(ysiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the nz, zsiz: number, size of grid cells (z direction)
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(nz) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set renderpar(zsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the ncat: number of thresholds / categories
    set line [gets $inf]
    set space [string first " " $line]
    set renderpar(ncat) [format "%d" [string range $line 0 $space]]

    ### reads the cat: thresholds / categories
    set line [gets $inf]
    for { set ii 1 } { $ii <= $renderpar(ncat) } { incr ii } {
	set space [string first " " $line]
	set renderpar(cat,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc read_volume_rendering_params { fname } {
    global renderpar
    global volume_rendering_box_w
    global datafl_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "parameters for volume_rendering" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for VOLUME_RENDERING" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in volume_rendering box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $volume_rendering_box_w "VOLUME RENDERING parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### reads name of datafl: selected file for volume rendering
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set renderpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set renderpar(datafl) [format "%s" $line]
    }
    $datafl_w configure -state normal -textbackground white -foreground black

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_volume_rendering_params { fname } {
    global tcl_platform
    global version
    global renderpar
    global volume_rendering_box_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### set file name in volume_rendering box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $volume_rendering_box_w "VOLUME RENDERING parameters   -   $fname_short"

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for VOLUME_RENDERING"
    puts $outf "*****************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: selected file for volume rendering
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $renderpar(datafl)]
    set line [format "%s" $renderpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\volume rendering file"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc select_zplane { plane } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global sim_select
    global 3d_select

    set cmdstr "geossav_3dview 1"
    set params3d(xy_nn) $plane
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    $visframe.subfr_1.mframe.ogl activate
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    if { $sim_select == 1 } {
	geossav_photo "dplane" 1 $plane 400 "XY"
    }
    update
}
#============================================================================


#############################################################################
proc select_yplane { plane } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global sim_select
    global 3d_select

    set cmdstr "geossav_3dview 1"
    set params3d(xz_nn) $plane
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    $visframe.subfr_1.mframe.ogl activate
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    if { $sim_select == 2 } {
	geossav_photo "dplane" 1 $plane 400 "XZ"
    }
    update
}
#============================================================================


#############################################################################
proc select_xplane { plane } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global sim_select
    global 3d_select

    set cmdstr "geossav_3dview 1"
    set params3d(yz_nn) $plane
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    $visframe.subfr_1.mframe.ogl activate
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    if { $sim_select == 3 } {
	geossav_photo "dplane" 1 $plane 400 "YZ"
    }
    update
}
#============================================================================


#############################################################################
proc change_to_xy_plane { } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global 3d_select

    $visframe.subfr_1.mframe.ogl activate
    set cmdstr "geossav_3dview 1"
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    geossav_photo "dplane" 1 1 400 "XY"
    update
}
#============================================================================


#############################################################################
proc change_to_xz_plane { } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global 3d_select

    $visframe.subfr_1.mframe.ogl activate
    set cmdstr "geossav_3dview 1"
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    geossav_photo "dplane" 1 1 400 "XZ"
    update
}
#============================================================================


#############################################################################
proc change_to_yz_plane { } {
    global renderpar
    global visframe
    global image_item
    global params3d
    global 3d_select

    $visframe.subfr_1.mframe.ogl activate
    set cmdstr "geossav_3dview 1"
    if { $3d_select(xy) } {
	append cmdstr " XY " $params3d(xy_nn)
    }
    if { $3d_select(xz) } {
	append cmdstr " XZ " $params3d(xz_nn)
    }
    if { $3d_select(yz) } {
	append cmdstr " YZ " $params3d(yz_nn)
    }
    eval $cmdstr
    $visframe.subfr_1.mframe.ogl swapbuffers

    geossav_photo "dplane" 1 1 400 "YZ"
    update
}
#============================================================================


#############################################################################
proc geossav_button1_press { x y } {
    global old_x old_y

    set old_x $x
    set old_y $y
}
#============================================================================


#############################################################################
proc geossav_button1_motion { x y } {
    global old_x old_y
    global sp
    global visframe
    global params3d
    global 3d_select

    set view_changed 0
    if { $old_x != $x } {
	set sp(longitude) [expr $sp(longitude) - (($x - $old_x)/5.0)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old_x $x
    }
    if { $old_y != $y } {
	set sp(latitude) [expr $sp(latitude) + (($y - $old_y)/5.0)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old_y $y
    }

    if { $view_changed } {
	$visframe.subfr_1.mframe.ogl activate
	set cmdstr "geossav_3dview 1"
	if { $3d_select(xy) } {
	    append cmdstr " XY " $params3d(xy_nn)
	}
	if { $3d_select(xz) } {
	    append cmdstr " XZ " $params3d(xz_nn)
	}
	if { $3d_select(yz) } {
	    append cmdstr " YZ " $params3d(yz_nn)
	}
	eval $cmdstr
	$visframe.subfr_1.mframe.ogl swapbuffers
    }
    
}
#============================================================================


#############################################################################
proc geossav_button3_press { x y } {
    global old3_x old3_y

    set old3_x $x
    set old3_y $y
}
#============================================================================


#############################################################################
proc geossav_button3_motion { x y } {
    global old3_x old3_y
    global sp
    global visframe
    global params3d
    global 3d_select

    set view_changed 0
    if { $old3_y != $y } {
	set diff [expr ($y - $old3_y)/50.0]
	if { $diff < -0.8 } {
	    set diff 0.8
	}
	set sp(dist) [expr $sp(dist)*(1.0 + $diff)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old3_y $y
    }

    if { $view_changed } {
	$visframe.subfr_1.mframe.ogl activate
	set cmdstr "geossav_3dview 1"
	if { $3d_select(xy) } {
	    append cmdstr " XY " $params3d(xy_nn)
	}
	if { $3d_select(xz) } {
	    append cmdstr " XZ " $params3d(xz_nn)
	}
	if { $3d_select(yz) } {
	    append cmdstr " YZ " $params3d(yz_nn)
	}
	eval $cmdstr

	$visframe.subfr_1.mframe.ogl swapbuffers
    }
    
}
#============================================================================

#
# End of proc section.
#
#############################################################################
proc geossav_volume_rendering_main { } {
    global exeidr
    source $exedir/Geossav_volume_rendering_globals.tcl
    
    . configure -cursor watch
    update
    
    set volume_rendering_box_w [toplevel .volume_rendering_box -class Dialog]
    wm resizable $volume_rendering_box_w false false
    wm sizefrom $volume_rendering_box_w program
    wm group $volume_rendering_box_w .
    wm transient $volume_rendering_box_w [winfo toplevel [winfo parent $volume_rendering_box_w]]
    wm title $volume_rendering_box_w "VOLUME RENDERING parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $volume_rendering_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set volume_rendering_tabnotebook_w [iwidgets::tabnotebook $volume_rendering_box_w.volume_rendering_tbn \
					    -width 5.4i \
					    -height 1.0i \
					    -tabpos n \
					    -tabforeground gray50 \
					    -background gray85 \
					    -bevelamount 2]
    
    pack $volume_rendering_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav VOLUME_RENDERING-parameters: Files (there is only one page)
    #
    set page [$volume_rendering_box_w.volume_rendering_tbn add \
		  -label "Files"]
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    # fields for rendering data
    set data_iframe_w [iwidgets::labeledframe $work_frame_w.data \
			   -labeltext "Rendering Input" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set data_frame_w [$data_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $data_frame_w.subfr1]
    
    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "Simulated realizations, file " \
		      -width 51 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable renderpar(datafl)]
    
    if { $renderpar(datafl) != {*.out} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".out" \
						    -filetypes {{{Simulation Files} {.out}} {{All Files} {.*}}} \
						    -initialfile $renderpar(datafl) \
						    -initialdir $exedir \
						    -title "Select simulated realizations file for VOLUME RENDERING"]
				 
				 ### select file name
				 if { $tmp_fname != {} } {
				     set renderpar(datafl) $tmp_fname
				     
				     ### set datafl widgets normal
				     if { $renderpar(datafl) == {*.out} } {
					 $datafl_w configure \
					     -state disabled -textbackground gray85 -foreground gray50
				     } elseif { $renderpar(datafl) != {*.out} } {
					 $datafl_w configure -state normal -textbackground white -foreground black
				     }
				     
				     ### create infofl path- and filename
				     set infofl_rootname [file rootname $renderpar(datafl)]
				     set infofl_dirname [file dirname $renderpar(datafl)]
				     set infofl_dirname_lenght [string length $infofl_dirname]
				     set infofl_name [string range $infofl_rootname \
							  [expr $infofl_dirname_lenght + 1] end]
				     set infofl_extension [file extension $renderpar(datafl)]
				     set info "/Info_"
				     regsub {} $infofl_extension $infofl_name infofl_1
				     regsub {} $infofl_1 $info infofl_2
				     regsub {} $infofl_2 $infofl_dirname infofl_3
				     set renderpar(infofl) $infofl_3
				     
				     ### open infofl and read data
				     if { $mofpar(infofl) != {} } {
					 read_volume_rendering_infofl_params $renderpar(infofl)
				     }
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    pack \
	$subframe1_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    pack \
	$data_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $volume_rendering_box_w.volume_rendering_tbn view "Files"    
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    
    set button_frame_w [frame $volume_rendering_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $mofpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select VOLUME RENDERING parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_volume_rendering_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       write_volume_rendering_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $renderpar(datafl) == {*.out} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "There is no rendering input file.\nYou should select a corresponding file first." \
					  -type ok
				  } elseif { $renderpar(datafl) != {*.out} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $renderpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}} \
							 -title "Save VOLUME RENDERING parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set param_fname $tmp_fname
					  write_volume_rendering_params $param_fname
				      }
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {

			      ### set the cursor on watch
			      . configure -cursor watch
			      update

			      ### messages, if there are no data
			      if { $renderpar(datafl) == {*.out} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "There is no rendering input file.\nYou should select a corresponding file first." \
				      -type ok
			      } elseif { $renderpar(datafl) != {*.out} } {

				  geossav_read $renderpar(datafl) \
				      $renderpar(nx) $renderpar(ny) $renderpar(nz) $renderpar(nsim) \
				      $renderpar(xsiz) $renderpar(ysiz) $renderpar(zsiz) 


				  set subframe1_w [frame $visframe.subfr_1]

				  set mframe [frame $subframe1_w.mframe]
				  set sp(longitude) -115.0
				  set sp(latitude) 60.0
				  set sp(dist) 3.0
				  geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)

				  opengl $mframe.ogl -width 500 -height 411 \
				      -rgb 1 \
				      -depth 1 \
				      -doublebuffer 1 \
				      -command "oglswap $mframe.ogl"
				  pack $mframe.ogl -side top
				  update

				  image create photo "dplane"
				  geossav_photo "dplane" 1 1 400 "XY"

				  set rframe [frame $subframe1_w.rframe]
				  set label [canvas $subframe1_w.rframe.label -width 500 -height 411]
				  set image_item(img) [$subframe1_w.rframe.label create image 0 0 \
							   -image "dplane" -anchor nw]

				  set image_item(arrow) [$subframe1_w.rframe.label create line \
							     $unitarrow(x0) $unitarrow(y0) $unitarrow(x1) $unitarrow(y1) \
							     -arrow both -fill red]
				  set image_item(text) [$subframe1_w.rframe.label create text \
							    [expr ($unitarrow(x0) + $unitarrow(x1))/2] 20 \
							    -text $unitarrow(unit) -fill red]

				  pack $label -side top
				  pack $mframe $rframe -side left -anchor nw
				  update

				  pack $subframe1_w
				  update

				  $mframe.ogl activate


				  set subframe2_w [frame $visframe.subfr_2]

				  set leftframe [frame $subframe2_w.leftframe]

				  set xy_checkbutton_w [checkbutton $leftframe.xy_cbutton \
							    -text "XY-plane" \
							    -variable 3d_select(xy) \
							    -command {}]

				  set xz_checkbutton_w [checkbutton $leftframe.xz_cbutton \
							    -text "XZ-plane" \
							    -variable 3d_select(xz) \
							    -command {}]
				  
				  set yz_checkbutton_w [checkbutton $leftframe.yz_cbutton \
							    -text "YZ-plane" \
							    -variable 3d_select(yz) \
							    -command {}]

				  
				  set middleframe [frame $subframe2_w.mframe]

				  set xy_pselect [scale $middleframe.xy_pselect \
						      -orient horizontal \
						      -showvalue true \
						      -length 300 \
						      -width 5 \
						      -from 1 -to $renderpar(nz) \
						      -command select_zplane]

				  set xz_pselect [scale $middleframe.xz_pselect \
						      -orient horizontal \
						      -showvalue true \
						      -length 300 \
						      -width 5 \
						      -from 1 -to $renderpar(ny) \
						      -command select_yplane]

				  set yz_pselect [scale $middleframe.yz_pselect \
						      -orient horizontal \
						      -showvalue true \
						      -length 300 \
						      -width 5 \
						      -from 1 -to $renderpar(nx) \
						      -command select_xplane]
				  

				  set rightframe [frame $subframe2_w.rightframe]

				  set rb_xy [radiobutton $rightframe.xy_plane \
						 -variable sim_select -value 1 \
						 -command change_to_xy_plane]
				  set rb_xz [radiobutton $rightframe.xz_plane \
						 -variable sim_select -value 2 \
						 -command change_to_xz_plane]
				  set rb_yz [radiobutton $rightframe.yz_plane \
						 -variable sim_select -value 3 \
						 -command change_to_yz_plane]
				  set sim_select 1


				  grid columnconfigure $leftframe 0 -minsize 275
				  grid columnconfigure $middleframe 1 -minsize 500
				  grid columnconfigure $rightframe 2 -minsize 275

				  grid $xy_checkbutton_w -row 1 -column 0 -sticky se -pady 0 -padx 8
				  grid $xz_checkbutton_w -row 2 -column 0 -sticky se -pady 7 -padx 8
				  grid $yz_checkbutton_w -row 3 -column 0 -sticky se -pady 0 -padx 8
				  grid $xy_pselect -row 0 -column 1 -sticky we -pady 0 -padx 8
				  grid $xz_pselect -row 1 -column 1 -sticky we -pady 0 -padx 8
				  grid $yz_pselect -row 2 -column 1 -sticky we -pady 0 -padx 8
				  grid $rb_xy -row 0 -column 2 -sticky sw -pady 0 -padx 8
				  grid $rb_xz -row 1 -column 2 -sticky sw -pady 7 -padx 8
				  grid $rb_yz -row 2 -column 2 -sticky sw -pady 0 -padx 8


				  pack \
				      $leftframe \
				      $middleframe \
				      $rightframe \
				      -side left -anchor w


				  bind $mframe.ogl <ButtonPress-1> {
				      geossav_button1_press %x %y
				  }
				  bind $mframe.ogl <B1-Motion> {
				      geossav_button1_motion %x %y
				  }
				  bind $mframe.ogl <ButtonPress-3> {
				      geossav_button3_press %x %y
				  }
				  bind $mframe.ogl <B3-Motion> {
				      geossav_button3_motion %x %y
				  }


				  pack $subframe2_w

				  set private(button) 1
      			      }

			      ### reset the cursor on arrow 
			      . configure -cursor arrow
			      update
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $volume_rendering_box_w
    
#    tkwait visibility $volume_rendering_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $volume_rendering_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $volume_rendering_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    update
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $volume_rendering_box_w <Destroy> {}
	destroy $volume_rendering_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }   
    return
}
#============================================================================
#=END=OF=VOLUME_RENDERING====================================================
