
# Copyright (c) 2001-2013, Scott D. Peckham

#---------------------------------------------------------------------
#
#   get_test_data_info()     # (2/22/12)
#   unit_test()
#   unit_test2()
#   get_neighbors_test()     # (9/14/11)
#
#---------------------------------------------------------------------
def get_test_data_info( data_name='TREYNOR' ):

    #------------------------------------------------------
    # Get name of platform (Darwin, Linux, Windows, Java)
    #------------------------------------------------------
    os_name = platform.system()

    #------------------------------------------------
    # Example of DEM with fixed-length pixels (UTM)
    #
    # NOTE!  Treynor DEM has NO FLATS, so doesn't
    #        provide a test for link_flats().
    #
    #     min(da) = 900.000  [m^2]
    #     max(da) = 900.000  [m^2]
    #     min(A)  =   0.000  [km^2]
    #     max(A)  = 0.49860  [km^2]
    #-----------------------------------------------
    if (data_name.upper() == 'TREYNOR'):
        cfg_prefix    = 'Treynor'
        if (os_name == 'Darwin'):
            cfg_directory = '/Applications/Erode/Data/Treynor_Iowa/'
        else:
            cfg_directory = '/home/csdms/models/erode/0.5/share/data/Treynor_Iowa/'

    #------------------------------------------------------
    # Example of DEM with fixed-angle pixels (Geographic)
    #     min(da) = 6802.824074169645  [m^2]
    #     max(da) = 6837.699120083246  [m^2]
    #     min(A)  =    0.000000000000  [km^2]
    #     max(A)  =  807.063354492188  [km^2]
    #------------------------------------------------------
    if (data_name.upper() == 'KY_SUB'):
        cfg_prefix    = 'KY_Sub'
        if (os_name == 'Darwin'):
            cfg_directory = '/Applications/Erode/Data/KY_Sub/'
        else:
            cfg_directory = '/home/csdms/models/erode/0.5/share/data/KY_Sub/'

    #------------------------------------------------
    # Example of DEM with fixed-length pixels (UTM)
    #     min(da) = 900.000  [m^2]
    #     max(da) = 900.000  [m^2]
    #     min(A)  =   0.000000000000  [km^2]
    #     max(A)  = 681.914184570312  [km^2]
    #-----------------------------------------------
    if (data_name.upper() == 'BEAVER'):
        cfg_prefix = 'Beaver'
        if (os_name == 'Darwin'):
            cfg_directory = '/Applications/Erode/Data/Beaver_Creek_KY/'
        else:
            cfg_directory = '/home/csdms/models/erode/0.5/share/data/Beaver_Creek_KY/'

    #------------------------------------------------
    # DEM generated by midpoint replacement method.
    # Now used only by "get_neighbors_test()".
    #------------------------------------------------
    if (data_name.upper() == '100X100'):
        cfg_prefix = 'Test'
        if (os_name == 'Darwin'):
            cfg_directory = '/Applications/Erode/Data/Random_100x100/'
        else:
            cfg_directory = '/home/csdms/models/erode/0.5/share/data/Random_100x100/'
            
    #-----------------------------------
    return cfg_directory, cfg_prefix

#   get_test_data_info()
#-----------------------------------------------------------------------
def unit_test( data_name='TREYNOR', SILENT=False, REPORT=True):

    #------------------------------------------------------
    # Note: data_name can be TREYNOR, KY_SUB or BEAVER.
    #----------------------------------------------------------------
    # Note:  This unit test computes D8 flow code and area grids
    #        as computed by the "d8_global" class to those computed
    #        by the "d8_local" class and then compares both of them
    #        to results obtained from RiverTools 3.0.
    #----------------------------------------------------------------
    # NOTE! The tests will appear to fail if the existing flow
    #       grid used for comparison was computed using a flat
    #       resolution method other than "Iterative linking".
    #
    # The KY_Sub and Beaver DEMs were processed using RiverTools
    # 3.0 using the WGS_1984 ellipsoid model for computing lengths
    # and areas.  The "Iterative linking" method was used for both
    # as the "Flat resolution method", to make them comparable to
    # the ones generated by functions in d8_base.py and
    # fill_pits.py.  Older versions of these data sets used other
    # methods and can't be compared directly.
    #
    # Make sure that LINK_FLATS=True, LR_PERIODIC=False, and
    # TB_PERIODIC=False in CFG file.
    #
    # NB! There is another "local" test in update_area_grid().
    #
    # NB! "d8.A_units" is read from CFG file "*_d8.cfg" and needs
    #     to be km^2 whenever RT3_TEST is used.
    #----------------------------------------------------------------

    #----------------------------------------------
    # Get cfg directory and prefix for unit tests
    #----------------------------------------------
    cfg_directory, cfg_prefix = get_test_data_info( data_name )
    
    #--------------------------
    # Change to CFG directory
    #--------------------------
    os.chdir( cfg_directory )
    
    #-----------------------------------
    # Start with a d8_global component
    #-----------------------------------
    import d8_global
    start1 = time.time()
    d8_glo = d8_global.d8_component()
    d8_glo.CCA    = False
    d8_glo.DEBUG  = True
    d8_glo.SILENT = SILENT
    d8_glo.REPORT = REPORT
    print 'Starting d8_global calculations >>>>>>>>>>>>>>>>>>>>>'
    d8_glo.initialize(cfg_prefix=cfg_prefix, mode="main",
                      SILENT=SILENT, REPORT=REPORT)
    d8_glo.RT3_TEST = True # (compare flow and area grid to RT3)
    d8_glo.update(SILENT=SILENT, REPORT=REPORT)
    print '------------------------------------------------------------'
    print 'Run time for d8_global =', (time.time() - start1), ' [secs]'
    print '------------------------------------------------------------'

    #--------------------------------------
    # Save D8 grids specified in CFG file
    #--------------------------------------
    d8_glo.code_gs_file = 'Global_' + d8_glo.code_gs_file
    d8_glo.area_gs_file = 'Global_' + d8_glo.area_gs_file
    d8_glo.dw_gs_file   = 'Global_' + d8_glo.dw_gs_file
    d8_glo.ds_gs_file   = 'Global_' + d8_glo.ds_gs_file
    d8_glo.save_grids()
    
    #--------------------------------------
    # Now initialize a d8_local component
    #--------------------------------------
    start2 = time.time()
    d8_loc = d8_component()
    d8_loc.CCA    = False
    d8_loc.DEBUG  = True
    d8_loc.SILENT = SILENT
    d8_loc.REPORT = REPORT
    print ' '
    print 'Starting d8_local calculations >>>>>>>>>>>>>>>>>>>>>'
    d8_loc.initialize(cfg_prefix=cfg_prefix, mode="driver",
                      SILENT=SILENT, REPORT=REPORT)
    d8_loc.RT3_TEST = True  # (compare flow and area grid to RT3)
    d8_loc.AREA_GRID_WARNING = True    ###########
    d8_loc.update(SILENT=SILENT, REPORT=REPORT)
    print '------------------------------------------------------------'
    print 'Run time for d8_local =', (time.time() - start2), ' [secs]'
    print '-----------------------------------------------------------'

    #--------------------------------------
    # Save D8 grids specified in CFG file
    #--------------------------------------
    d8_loc.code_gs_file = 'Local_' + d8_loc.code_gs_file
    d8_loc.area_gs_file = 'Local_' + d8_loc.area_gs_file
    d8_loc.dw_gs_file   = 'Local_' + d8_loc.dw_gs_file
    d8_loc.ds_gs_file   = 'Local_' + d8_loc.ds_gs_file
    d8_loc.save_grids()
    
    #-----------------------------------------
    # Compare the results from the 2 methods
    #-----------------------------------------
    print 'Results of comparing d8_local to d8_global:'
    w  = where(d8_glo.d8_grid != d8_loc.d8_grid)
    nw = w[0].size
    if (nw != 0):
        print 'd8_grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: D8 code grids are identical.'
    #-------------------------------------------------
    w  = where(d8_glo.A != d8_loc.A)
    nw = w[0].size
    if (nw != 0):
        print 'A grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: A grids are identical.'
    #-------------------------------------------------
    w  = where(d8_glo.dw != d8_loc.dw)
    nw = w[0].size
    if (nw != 0):
        print 'dw grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: dw grids are identical.'
    #-------------------------------------------------
    w  = where(d8_glo.ds != d8_loc.ds)
    nw = w[0].size
    if (nw != 0):
        print 'ds grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: ds grids are identical.'
    #-------------------------------------------------
    w  = where(d8_glo.ID_grid != d8_loc.ID_grid)
    nw = w[0].size
    if (nw != 0):
        print 'ID_grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: ID_grids are identical.'
    #-------------------------------------------------------------
    w  = where(d8_glo.parent_ID_grid != d8_loc.parent_ID_grid)
    nw = w[0].size
    if (nw != 0):
        print 'parent_ID_grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: parent_ID_grids are identical.'

    #-----------------------------------------------------------------------
    # (1/25/12) Removed noflow_IDs since it wasn't used by either
    # erode_d8_global.py or erode_d8_local.py and affected performance.
    #-----------------------------------------------------------------------
    # (9/20/11) In these next several, w is usually a tuple with one
    # element, i.e. w = (array([], dtype=int32),).  However, in some
    # cases we can get "w = ()", or an empty tuple. (Not sure why, yet.)
    # When that happens, we get the error message:
    #     "IndexError: tuple index out of range".
    # However, "size(w)" returns 0 in this case. Since we are only
    # interested in whether nw is zero, we use "nw = size(w)" here
    # instead of "nw = w[0].size", which sometimes fails. 
    #-----------------------------------------------------------------------
##    w  = where( logical_or(d8_glo.noflow_IDs[0] != d8_loc.noflow_IDs[0],
##                           d8_glo.noflow_IDs[1] != d8_loc.noflow_IDs[1]))
##    # print 'w =', w, '###############################'
##    nw = size(w)
##    if (nw != 0):
##        print 'noflow_IDs grids differ at', nw, 'locations.'
##    else:
##        print 'SUCCESS: noflow_IDs are identical.'
    #-----------------------------------------------------------------------
    w  = where( logical_or(d8_glo.w1[0] != d8_loc.w1[0],
                           d8_glo.w1[1] != d8_loc.w1[1]))
    nw = size(w)
    if (nw != 0):
        print 'w1 IDs differ at', nw, 'locations.'
    else:
        print 'SUCCESS: w1 IDs are identical.'
    #-----------------------------------------------------------------------
    w  = where( logical_or(d8_glo.p1[0] != d8_loc.p1[0],
                           d8_glo.p1[1] != d8_loc.p1[1]))
    nw = size(w)
    if (nw != 0):
        print 'p1 IDs grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: p1 IDs are identical.'
    #-----------------------------------------------------------------------
    w  = where( logical_or(d8_glo.w2[0] != d8_loc.w2[0],
                           d8_glo.w2[1] != d8_loc.w2[1]))
    nw = size(w)
    if (nw != 0):
        print 'w2 IDs differ at', nw, 'locations.'
    else:
        print 'SUCCESS: w2 IDs are identical.'
    #-----------------------------------------------------------------------
    w  = where( logical_or(d8_glo.p2[0] != d8_loc.p2[0],
                           d8_glo.p2[1] != d8_loc.p2[1]))
    nw = size(w)
    if (nw != 0):
        print 'p2 IDs grids differ at', nw, 'locations.'
    else:
        print 'SUCCESS: p2 IDs are identical.'
    #-----------------------------------------------------------------------
    print 'Finished with unit_test().'
    print ' '
                
#   unit_test()
#-----------------------------------------------------------------------
def unit_test2( data_name='TREYNOR', test_name='SQUARE',
                SILENT=False, REPORT=True ):

    #-------------------------------------------------------------
    # Note: data_name can be:  TREYNOR, KY_SUB or BEAVER.
    #       test_name can be:  STRIP, PIXEL or SQUARE.
    #-------------------------------------------------------------
    # Note: This unit test sets area grid values to zero for
    #       different regions of cells in the DEM to see if
    #       d8_local can "fix" them and all affected downstream
    #       cells. The following binary grid files are generated
    #       and can be displayed as color images to help with
    #       testing:
    #           (1) [cfg_prefix]_A-perturb.rtg
    #           (2) [cfg_prefix]_A-perturb2.rtg
    #           (3) [cfg_prefix]_A-after.rtg.
    #       The first one is an area grid, created with a
    #       prior call to d8_local.update(), but with A=0 for
    #       a set of "perturbed" cells, such as a big square.
    #       The 2nd one is similar, but also shows all of the
    #       downstream cells whose area values need to be
    #       updated when values at the perturbed cells change.
    #       The 3rd one shows the resulting area grid after
    #       new area values have been computed for all affected
    #       cells.
    #-------------------------------------------------------------
    cfg_directory, cfg_prefix = get_test_data_info( data_name )

    #--------------------------
    # Change to CFG directory
    #--------------------------
    os.chdir( cfg_directory )
    
    #----------------------------------
    # Initialize a d8_local component
    #----------------------------------
    start  = time.time()
    d8_loc = d8_component()
    d8_loc.CCA    = False
    d8_loc.DEBUG  = True
    d8_loc.SILENT = SILENT
    d8_loc.REPORT = REPORT
    d8_loc.initialize(cfg_prefix=cfg_prefix, mode="driver",
                      SILENT=SILENT, REPORT=REPORT)

    #------------------------------------------------------
    # We call update() here in order to get results from
    # d8_local before perturbation, to compare. (2/22/12)
    #------------------------------------------------------
    d8_loc.RT3_TEST = True # (compare flow and area grid to RT3)
    d8_loc.update(SILENT=SILENT, REPORT=REPORT)
    orig_A  = d8_loc.A.copy()        # (save results)
    orig_fg = d8_loc.d8_grid.copy()
    print '------------------------------------------------------------'
    print 'Run time for d8_local =', (time.time() - start), ' [secs]'
    print ' (unperturbed case) '
    print '-----------------------------------------------------------'
    print ' '
    
    #---------------------------------------------------
    # Set IDs corresponding to some region in the DEM,
    # call update() and compare results, especially to
    # check update_area_grid().
    #---------------------------------------------------
    ID_col = (d8_loc.nx / 2)
    ID_row = (d8_loc.ny / 2)
    ID     = (ID_row * d8_loc.nx) + ID_col
        
    print '---------------------------------'
    print 'Starting perturbation test...'
    print '---------------------------------'
    if (test_name == 'STRIP'):
        print '>>>> Perturbing a strip near center...'
        width  = 15
        height = 2
        #-----------
        ID_col = ID_col - (width/2)  # (for centering)
        ID     = (ID_row * d8_loc.nx) + ID_col
        ramp = numpy.arange( width )
        ID_list = numpy.zeros( width * height, dtype='int32' )
        for k in xrange( height ):
            ID_list[k*width : (k+1)*width] = ID + ramp
            ID += d8_loc.nx   # (skip down a row)
    elif (test_name == 'PIXEL'):
        #--------------------------------------------------------
        # Single-pixel test that identified a subtle bug, found
        # and fixed on (2/16/12).  All downstream pixels, which
        # can be seen in the test file [prefix]*-perturb2.rtg,
        # must be set to zero.
        #--------------------------------------------------------
        print '>>>> Perturbing center pixel...'
        ID_list = numpy.array( [ID] )
    elif (test_name == 'SQUARE'):
        print '>>>> Perturbing big square near center...'
        side = 20  # (side length of square)
        #----------
        ID_row = ID_row - (side/2)       # (for centering)
        ID     = (ID_row * d8_loc.nx) + ID_col
        ramp = numpy.arange( side )
        ramp = ramp - (side/2)        # (for centering)
        ID_list = numpy.zeros( side * side, dtype='int32' )
        for k in xrange(side):
            ID_list[k*side : (k+1)*side] = ID + ramp
            ID += d8_loc.nx   # (skip down a row)

    #------------------------------------------
    # Do we need this to make sure everything
    # is recomputed ? (2/23/12)
    #------------------------------------------
    d8_loc.d8_grid.flat[ ID_list ] = 0
    
    #------------------------------------------------
    # Call update on the d8_local component (again)
    #------------------------------------------------
    d8_loc.IDs               = ID_list
    d8_loc.PERTURB_TEST      = True
    d8_loc.AREA_GRID_WARNING = True  # (2/22/12)
    d8_loc.RT3_TEST          = True  # (compare flow and area grid to RT3)
    d8_loc.update(SILENT=SILENT, REPORT=REPORT)
    #---------------------------------------------------------------
    w  = where( d8_loc.d8_grid != orig_fg )
    nw = w[0].size
    print 'Results of perturbation test:'
    print '>>>> Number of perturbed IDs =', ID_list.size
    if (nw != 0):
        print '>>>> d8_grids differ from original at', nw, 'locations.'
    else:
        print '>>>> SUCCESS: d8_grids are identical to original.'
    #---------------------------------------------------------------
    w  = where( d8_loc.A != orig_A )
    nw = w[0].size
    if (nw != 0):
        print '>>>> A grids differ from original at', nw, 'locations.'
    else:
        print '>>>> SUCCESS: A grids are identical to original.'    
    #---------------------------------------------------------------
    print 'Finished with unit_test2().'
    print ' '
    
#   unit_test2()
#-----------------------------------------------------------------------
def get_neighbors_test(data_name='100x100'):

    #----------------------------------------------
    # (9/14/11) Test to make 100% sure that the
    # get_neighbors() function works as intended.
    #----------------------------------------------
    # To check results against comments, we need
    # data_name to be '100x100'.
    #----------------------------------------------
    d8_loc = d8_component()

    d8_loc.CCA    = False
    d8_loc.REPORT = False
    d8_loc.DEBUG  = False
    d8_loc.SILENT = True

    cfg_directory, cfg_prefix = get_test_data_info( data_name )

    #---------------------------
    # Change to cfg_directory.
    # Moved here on: 11/15/11.
    #---------------------------
    os.chdir( cfg_directory )
    
    d8_loc.initialize( cfg_prefix=cfg_prefix, mode="driver" )
    nx = d8_loc.nx
    ny = d8_loc.ny
    print ' '
    print '(nx, ny) =', nx, ny
    print ' '
    
    #-----------------------------
    # Corner tests (3 neighbors)
    #-----------------------------
    ID = numpy.array([0])
    UL_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([ny-1])
    UR_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([(ny-1)*nx])
    LL_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([(nx*ny) - 1])
    LR_nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (four corners) ='
    #-----------------------------------------
    print '   ', UL_nbr_IDs, '    (upper-left corner)'
    print 'vs. [  1 100 101]'
    #-----------------------------------------
    print '   ', UR_nbr_IDs, '    (upper-right corner)'
    print 'vs. [ 98 198 199]'
    #-----------------------------------------
    print '   ', LL_nbr_IDs, ' (lower-left corner)'
    print 'vs. [9800 9801 9901]'
    #-----------------------------------------
    print '   ', LR_nbr_IDs, ' (lower-right corner)'
    print 'vs. [9898 9899 9998]'
    print ' '
    
    #---------------------------
    # Edge tests (5 neighbors)
    #---------------------------
    ID = numpy.array([1])
    T_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([100])
    L_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([199])
    R_nbr_IDs = d8_loc.get_neighbors( ID )
    ID = numpy.array([9901])
    B_nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (along edges) ='
    #---------------------------------------------
    print '   ', T_nbr_IDs, '      (top edge)'
    print 'vs. [  0   2 100 101 102]'
    #-----------------------------------------
    print '   ', L_nbr_IDs, '      (left edge)'
    print 'vs. [  0   1 101 200 201]'
    #-----------------------------------------
    print '   ', R_nbr_IDs, '      (right edge)'
    print 'vs. [ 98  99 198 298 299]'
    #-----------------------------------------
    print '   ', B_nbr_IDs, ' (bottom edge)'
    print 'vs. [9800 9801 9802 9900 9902]'
    print ' '

    #----------------------------------------------
    # Interior test:  [0 1 2 100 102 200 201 202]
    #----------------------------------------------
    ID = numpy.array([101])
    nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (interior) ='
    print '   ', nbr_IDs
    print 'vs. [  0   1   2 100 102 200 201 202]'
    print ' '

    ######################################################
    # (9/14/11) Found and fixed a bug on next few cases.
    ######################################################

    #-------------------------------
    # Interior, adjacent pair test
    #------------------------------------
    # [0 1 2 3 100 103 200 201 202 203]
    #------------------------------------
    ID = numpy.array([101, 102])
    nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (interior, adjacent pair) ='
    print '   ', nbr_IDs
    print 'vs. [  0   1   2   3 100 103 200 201 202 203]'
    print ' '
    
    #-------------------------------
    # Interior, disjoint pair test
    #----------------------------------------------
    # [0 1 3 2 4 100 102 104 200 201 203 202 204]
    #----------------------------------------------
    ID = numpy.array([101, 103])
    nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (interior, disjoint pair) ='
    print '   ', nbr_IDs
    print 'vs. [  0   1   2   3   4 100 102 104 200 201 202 203 204]'
    print ' '
    
    #---------------------------------
    # Interior, adjacent triple test
    #--------------------------------------------
    # [0 1 2 3 100 103 200 301 201 302 203 303]
    #--------------------------------------------
    ID = numpy.array([101, 102, 202])
    nbr_IDs = d8_loc.get_neighbors( ID )
    print 'nbr_IDs (interior triple) ='
    print '   ', nbr_IDs
    print 'vs. [  0   1   2   3 100 103 200 201 203 301 302 303]'
    print ' '
    
#   get_neighbors_test()
#---------------------------------------------------------------------


