/*! \file stage_inp.c
\brief Establish linkage between two grids, map surface water depth (misnamed stage) input data.

	This has a single function.\n
  
Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Dec 2004 \n
*/
	
/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Doxygen tags added 
	
*/
#include "stage.h"


/* \brief Get spatial input time series of water depth (NOT stage)

	Using the grid_map functions, this returns surface water depth, at ELM (i.e., SME) grid scale, for ELM domain boundary cells.
	This function was originally **intended** to return stage data (m NGVD '29, incl. ELM base datum depth) at ELM (SME) grid scale.
	However, this current version (ELMv2.3, Nov '04) uses the SFWMM (positive/negative) water depth relative to a SFWMM grid cell elevation (ft NGVD'29),
	converts it to the ELM grid scale, and converts that value to meters of water depth relative to that land surface elevation.
	This is (stage - land_surface_elev = water depth).
	
	\param stageSME Depth (stage misnamed) data array (m) at the ELM/SME grid scale
	\return success 	*/
int stage_data_wmm(float* stageSME)           /* array of float of size s0*s1 */
{

  int i,j,k;
  int success = 1, fail = -1;
  int stat = success;
  char gridmapfilename[135];

  if(SimTime.TIME==0)  {      
/* elm_OG_map is data structure containing the mapping attributes at two scales */
    elm_wmm_map = elm_OG_map;

    if(elm_wmm_map == NULL) {
      sprintf(msgStr, "Mapping grids and setting up stage data...");
      usrErr (msgStr);

      sprintf(gridmapfilename, "%s%s/Data/gridmapping.txt", ModelPath, ProjName );   
      stat = mapGrids(gridmapfilename);
      elm_wmm_map = elm_OG_map;
    }

    if(debug > 4) {
      printGridMap();
      sprintf(msgStr,"stage_data_wmm==> Finished mapping grids");
      usrErr (msgStr);
    }

    sprintf(stage_binfilename, "%s%s/Data/BoundCond_stage.BIN", ModelPath, ProjName );
    /* initializing data structures, move pointer to initial date (gridmap.c) */
    stat = initDataStruct(stage_binfilename,&stage_struct);  

    if(debug > 4) {
      /*printELM2Grid_io(); */
      /*drawELM2Grid_io(); */
      sprintf(msgStr,"stage_data_wmm==> Finished initializing");
      usrErr (msgStr); 
    }

  } /* end of SimTime.TIME=0 */
  
  
  if(stage_struct.day >= stage_struct.recRead) {      /* process the data in batch */
      sprintf(msgStr,"Processing batch of stage data...");
      usrErr (msgStr); 
      stat = processData(stage_binfilename,&stage_struct);

      if(debug > 4 ) {
        /*printBatchData(stageWMM,gridio_batch_len,widCnt);*/ /* TODO: remove this printBatchData function when sure is no longer needed */
        sprintf(msgStr,"stage_data_wmm==> Finished processing data");
        usrErr (msgStr); 
      }
  } /* end of if */


  if(stage_struct.day < stage_struct.recRead) {      /* pass the data day by day */
    returnData(stageSME,&stage_struct); 

    for(i = 0; i < s0; i++) {
      for(j = 0; j < s1; j++) {
        k = i*s1+j;
        stageSME[T((i+1),(j+1))] = stage_struct.dataELM[k] * conv_ftTOm;  /* convert data from feet to meters */
      }
    } 

    if(debug > 4) {
      sprintf(msgStr,"stage_data_wmm==> Finished returning data");
      usrErr (msgStr); 
    }

  } /* end of if */
    
  return success;
}


