/*! \file  Serial.c
\brief Utilities that can be different between serial & parallel codes.

This source file contains functions that are (or can be) different between
single-processor (serial) vs multi-processor implementations.  These functions 
are used in "utility" tasks that generally relate to I/O, set-up of the 
simulation environment, etc (and are generally called from Driver_Utilities.c). \n

Early versions of ELM had an analagous "Parallel.c", Express.c, etc.
for code that required parallel-specfic routines.  (Note: in ELM v1.0,
the parallel coding efforts were merged w/ serial code (elsewhere), allowing 
completely generic compilation.  Complexities due to vector-raster 
integration of canals were solved, and performance was very good (up to ~8 procs). 
However, that code development was halted in ~2000, and we do not anticipate 
parallel implementations of ELM in the near-term (Dec 2004).   \n

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Dec 2004 \n
*/
	
/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Unused functions removed 
       		- Re-organized, clarified scopes, Doxygen tags added 
       		- Functionality same as v2.1/v2.2 
	
*/

#include "serial.h"

/*! \brief Read data for spatial (map) input binary file 
	\param filename data variable's name
	\param data variable's array data
	\return byte size of data values
	*/
byte readMap( char* filename, void* data)
{
  int gsize, nbytes; byte bsize;
  FILE* bFile;
  char binFileName[300];
  
  bsize = read_header(filename);

  sprintf(binFileName,"%s/%s/Data/Map_bin/%s",ModelPath,ProjName,filename);

  if((bFile = fopen(binFileName,"rb") ) == NULL ) { 
    fprintf(stderr,"Can't Open Mapfile: %s",binFileName); 
    exit(0); 
  }
  gsize = gbl_size[0] * gbl_size[1];
  nbytes = fread((char*)data, (int)bsize, gsize, bFile );
  if(debug) { 
    fprintf(dFile," %d of %d items Read for %s, size = %x bytes ",
	    nbytes,gsize,filename,bsize); 
  }
  fclose(bFile);
  return bsize;
}


/*! \brief Read header info for spatial (map) input binary file 
	\param filename data variable's file name
	\return byte size of data values
*/
int read_header(char* filename)
{
  FILE* file;
  char headerFileName[300];
  
  int rows, cols, i, Dsize=1, test, offset;
  char ch, dirCh;
  long int start;
	
  sprintf(headerFileName,"%s/%s/Data/Map_head/%s",ModelPath,ProjName,filename);

  file = fopen(headerFileName,"r");
  if(file==NULL) { 
    fprintf(stderr,"Unable to open header file %s.\n",headerFileName); 
    fflush(stderr); 
    exit(0);
  }
  else { 
    fprintf(dFile,"Successfully opened header file %s.\n",headerFileName);  
    fflush(dFile); 
  } 
  scan_forward(file,"cols:");
  fscanf(file,"%d",&cols);
  fprintf(dFile,"cols = %d\n",cols); 
  scan_forward(file,"rows:");
  fscanf(file,"%d",&rows);
  fprintf(dFile,"rows = %d\n",rows);
  fflush(dFile);
  if( !( rows == gbl_size[0] && cols == gbl_size[1] ) ) { 
    fprintf(stderr,"Error, Wrong map size: %s: (%d,%d) : (%d,%d)\n",
	     headerFileName,rows,gbl_size[0],cols,gbl_size[1]); 
  }
  start = ftell(file);
  test = scan_forward(file,"format:");
  if(test < 1) { 
    clearerr(file); 
    fseek(file, start, 0); 
  }
  else {	
    fscanf(file,"%d",&Dsize);
    fprintf(dFile,"size = %d\n",Dsize);
    Dsize = Dsize + 1; /* using header generated by GRASS, the byte size is actually the value read from header plus 1 */
    if( Dsize < 1 || Dsize > 4 ) {  
      fprintf(stderr,"Error, Illegal map size: %s  (%d)\n",headerFileName,Dsize); 
      exit(0); 
    }
  }
  fclose(file);
  return Dsize;
}

/*! \brief Write data for spatial (map) output binary file 
	\param filename Filename (variable name) of data 
	\param data Data array
	\param bSize Byte size of data
	\param type Type, or format, of (binary) data
	\param index Index used in HDF output (which is unused)
	*/
void writeMap(char* filename, void* data, int bSize, unsigned char type, int index)
{
  int gsize, istat;
  FILE* bFile; 
  
  if(type=='M') write_header(filename,bSize);
  
  if(type == 'H') {
#if HDF
    if(index==0) { istat = DFR8putimage(filename,(char*)data,s1,s0,0); }
    else { istat = DFR8addimage(filename,(char*)data,s1,s0,0); }
    if(istat != 0) printf("\nerror writing HDF file: %s\n",filename);
#endif
  } 
  else {
    if((bFile = fopen(filename,"wb") ) == NULL ) { 
      fprintf(stderr,"Can't Open Mapfile: %s",filename); 
      exit(0); 
    }
    gsize = gbl_size[0]*gbl_size[1]*bSize;
    fwrite((char*)data, gsize, 1, bFile );
    fclose(bFile);
  }
}

/*! \brief Write header info for spatial (map) output binary file 
	\param mapFileName Filename (variable name) of data 
	\param size Byte size of data
*/
void  write_header(char* mapFileName,int size)
{
  FILE* header;
  fprintf(dFile,"Writing Header: filen = %s\n",mapFileName); 
  fflush(dFile);
  header = fopen(mapFileName,"w");
  if(header==NULL) { 
    fprintf(stderr,"Can't open header file: %s\n",mapFileName); 
    fflush(stderr); exit(-1); 
  }
  fprintf(header,"ROWS=%d\nCOLUMNS=%d\nCELLSIZE=1", 
		  gbl_size[0], gbl_size[1]);
  fprintf(header,"\nUNITS=KM\nFORMAT=BIN\nSIZE=%d\nLOCATION=\"%s.bin\"\n",
		  size,mapFileName); /* prior to v2.3, had the file delimited by the fileDelimitChar */
  fclose(header);
  strcat(mapFileName,".bin");
}



/*! \brief (Unused). Read time series input data for spatial interpolations
	\param filename Filename of data 
	\param format Always float data format
	\param index Unused. Was used to concatenate multiple files over time
	\param Npt Number of time points
	\param TStep Time step
	\param col column ID (for different variables)
*/
float* readSeriesCol(char *filename, int format, int index, int* Npt, float* TStep, int col )
{
    int line = 0, cread=1, itest, j, sLen = 372 ;
    FILE *cfile;
    char  ctest, cmark[373], ret = '\n';
    unsigned char cmove[373], cmv=0;
    char tfilename[200], date_read[20],ss[200];
    int yyyy,mm, dd;
    double Jdate_read; 
    float  *tvar, *nullvar, first[10];

    sprintf(tfilename,"%s%d.ts",filename,index);
    if(debug>4) fprintf(dFile,"\nReading file %s, col %d, index = %d\n",tfilename,col,index); fflush(dFile);
    cfile = fopen(tfilename,"r"); 
    if(cfile==NULL) {  
 	if( index > 0 ) {  fprintf(stdout,"\nWARNING: Unable to open timeseries file %s, using %s0.ts\n",tfilename,filename); return 0; } 
	else { fprintf(stdout,"\nERROR: Unable to open timeseries file %s\n",tfilename); exit(-1); } 
    }

    if (format != 2) { fprintf(stderr,"ERROR: only using floats in read_timeseries: %d\n",format); exit(0); }
    
           sLen = PORnumday; /* read/determined in genericDriver */
            tvar = (float*) nalloc( (sLen+2)*sizeof(float), "timeseries temp" );
            scan_forward(cfile,"{END COMMENT}\n");
            fgets(ss,200,cfile); /* skip  the header line */
            
            do {
                fscanf(cfile, "%d %d %d %f %f %f %f", &yyyy,&mm,&dd,&first[1],&first[2],&first[3],&first[4] ); 
        /* julian day, returns a double (args = month, day, year, hours, minutes, seconds) */
                Jdate_read = julday(mm, dd, yyyy, 0, 0, 0); 
                if (Jdate_read > Jdate_init) {printf (" \n***Error\nInit date in %s file > simulation start date.\n",tfilename); exit (-1);}
                if (feof(cfile)) {
	printf (" \n***Error\nNon-matching starting date for %s meteorological time series\n",tfilename); 
			exit (-1);
		}
            } while (Jdate_read < Jdate_init);
            
            tvar[0] = first[col]; /* have now read the first data record, assign to array */
            if(debug>4) fprintf(dFile,"%d %d %d %f\n",yyyy,mm,dd,tvar[0]);
            for (line = 1; line<sLen; line++) {
            
            /* each case for reading a different column */
                switch (col) {
                    case 1:
                        itest = fscanf(cfile,"%d %d %d %f %f %f %f",&yyyy,&mm,&dd, &(tvar[ line ]),&nullvar, &nullvar, &nullvar);
                        break;
                    case 2:
                        itest = fscanf(cfile,"%d %d %d %f %f %f %f",&yyyy,&mm,&dd,&nullvar, &(tvar[ line ]), &nullvar, &nullvar);
                        break;
                    case 3:
                        itest = fscanf(cfile,"%d %d %d %f %f %f %f",&yyyy,&mm,&dd,&nullvar, &nullvar, &(tvar[ line ]), &nullvar);
                        break;
                    case 4:
                        itest = fscanf(cfile,"%d %d %d %f %f %f %f",&yyyy,&mm,&dd,&nullvar, &nullvar, &nullvar, &(tvar[ line ]));
                        break;
                    default:
                        printf ("\nError in interpolation time series data.\n"); 
                        exit(-1);
                        break;
                } /* end switch */

                if(debug>4) fprintf(dFile,"%d %d %d %f\n",yyyy,mm,dd,tvar[ line ]);
                if (feof(cfile)) {printf (" \n***Error\nExceeded number of records for %s meteorological time series\n",tfilename); exit (-1);}
            }
            tvar[ sLen ] = *TStep;
 
    if(debug>4) { fprintf(dFile,"\nDone Reading file %s\n",tfilename); fflush(dFile); }
    
    fflush(stdout);
    fclose(cfile);
    *Npt = sLen;
    return tvar;
}




/*! \brief Read the input data file to get a habitat-specfic parameter array

	This gets an array of the targeted parameter, a value for each of (usually)
	multiple habitats.
	\param s_parm_name Name of the sensitivity parameter being varied for this run
	\param s_parm_relval Indicator of current value of relative range in sensitivity parm values: nominal (0), low (1), or high (2) 
	\param parmName String that identifies the parameter to be read in dbase file
	\return fTmp Array of values of single parameter across multiple habitats
*/
float* get_hab_parm( char* s_parm_name, int s_parm_relval, char* parmName ) 
{
	int hIndex, end=0;
	FILE* parmFile;
	float* fTmp;
	char lline[2001], *line;
	int pLoc = -1;
	int foundParm= False;
	char parmNameHead[30];
	char modelFileName[300];
    char HabParmFile[300];
    char *fileAppend;
    extern ProgAttr *ProgExec;
    
	
	if ( (fTmp = (float*) malloc( MAX_NHAB * sizeof(float) ) ) == NULL) {
        sprintf(msgStr, "Failed to allocate memory for HabParm array.\n ") ;
        usrErr(msgStr);
        exit( -2 ) ;
	};
	fTmp[0] = 0.0;

    fileAppend = match_Sparm( s_parm_name, s_parm_relval, parmName);
	sprintf(HabParmFile,"HabParms_%s",fileAppend);
	sprintf(modelFileName,"%s/%s/Data/%s",ModelPath,ProjName,HabParmFile);
	parmFile = fopen(modelFileName,"r");
	if(parmFile==NULL) {fprintf(stdout,"\nERROR: Unable to open dBase file %s\n",modelFileName); exit(-1); }
	
     fgets(lline, 2000, parmFile); /* skip 1st header line  */
     fgets(lline, 2000, parmFile); /* read 2nd header line with column names */
     line=lline;

     while (!foundParm)  
     {
     	sscanf( line,"%s",&parmNameHead);
     	if (strcmp(parmName,parmNameHead) ==0) foundParm = True;
     	pLoc++;
    	if (*line == EOS || *line == EOL) {
    	   sprintf(msgStr,"ERROR: Could not find header tag of %s in HabParms dbase file.", parmName); 
    	   usrErr(msgStr); 
    	   exit(-1);
    	}
        line = Scip( line, TAB );
     }
    
    sprintf(msgStr,"Parameter group: %s",parmName); 
    fflush(dFile); 
    WriteMsg(msgStr,1); 
	for( hIndex = 1; hIndex < MAX_NHAB; hIndex++) {
		goto_index (parmFile, gHabChar, hIndex); 
		fTmp[hIndex] = get_Nth_parm( parmFile, pLoc, &end, hIndex );
		if(end) break;
	}
	fclose(parmFile);
	habNumTot = hIndex-1;

	fflush(dFile); 
	/* TODO: this HARDCODES into the BIR (BIRavg) model output file ONLY, 
	the incorrect impression of running sensitivity on a single habitat ID 
	(hab 2 = sawgrass plain, ELMv2.4) */
    if (strcmp(s_parm_name,parmName) == 0) ProgExec->S_ParmVal = fTmp[2];
	return fTmp; 
}

/*! \brief Determine if the parameter is being varied for a sensitivity analysis.

	\param s_parm_name Name of the sensitivity parameter being varied for this run
	\param s_parm_relval Indicator of current value of relative range in sensitivity parm values: nominal (0), low (1), or high (2) 
	\param parmName String that identifies the parameter to be read in dbase file
	\return fileAppend The suffix (not extension) to be appended to end of base name of parameter input file.
*/
char* match_Sparm( char* s_parm_name, int s_parm_relval, char* parmName) 
{
    char *fileAppend;
 
    if (strcmp(s_parm_name,parmName) == 0) /* match, it is a parameter being varied in this set of runs */
    	{ 
    		switch (s_parm_relval) {
    		   case 0: fileAppend = "NOM"; break;
    		   case 1: fileAppend = "LO"; break;
    		   case 2: fileAppend = "HI"; break;
    		   default:  {
    		      sprintf(msgStr,"ERROR: The relative parameter value (%d) is unknown for %s",s_parm_relval, s_parm_name); 
    		      usrErr(msgStr);
    		      exit(-1);
	           }
           }
        }
    else
    	{ /* no match, it is NOT a parameter being varied in this set of runs */
	      fileAppend = "NOM"; 
	    }
    return (fileAppend);
}


/*! \brief Read the input data file to get a global parameter

	This gets the single global value of the targeted parameter.
	\param s_parm_name Name of the sensitivity parameter being varied for this run
	\param s_parm_relval Indicator of current value of relative range in sensitivity parm values: nominal (0), low (1), or high (2) 
	\param parmName String that identifies the parameter to be read in dbase file
	\return parmVal The value of the global parameter
*/
float get_global_parm( char* s_parm_name, int s_parm_relval, char* parmName) 
{
  int test;
  char modelFileName[300];
  FILE *parmFile;
  float parmVal;
  char parmNameHead[30];
  extern ProgAttr *ProgExec;


  char GlobalParmFile[50];
  char *fileAppend;
  
  fileAppend = match_Sparm( s_parm_name, s_parm_relval, parmName);
  sprintf(GlobalParmFile,"GlobalParms_%s",fileAppend);
  sprintf(modelFileName,"%s/%s/Data/%s",ModelPath,ProjName,GlobalParmFile);
  
  parmFile = fopen(modelFileName,"r");
  if(parmFile==NULL) { 
  	sprintf(msgStr,"ERROR, can't open data file %s",modelFileName); 
    usrErr(msgStr);
  	Exit(0); 
  }

  sprintf(parmNameHead,"%s=", parmName);
  scan_forward(parmFile,parmNameHead);
  if ( (test = fscanf(parmFile,"%f",&parmVal) ) <= 0 ) { 
    sprintf(msgStr,"ERROR in reading %s from %s; see Driver0.out debug file.", parmName, modelFileName); 
    usrErr(msgStr);
    Exit(0); 
  }
  sprintf(msgStr,"%s %f\n",parmNameHead, parmVal); 
  WriteMsg(msgStr,1); 

  fclose(parmFile);
  if (strcmp(s_parm_name,parmName) == 0) ProgExec->S_ParmVal = parmVal;
  return parmVal;
}

/*! \brief Point time series output: open files and print headers.  

	Provides for multi-column (locations), multi-file (model variables)format of point
	time series.  The user can define (via Model.outList commands), for any variable in the model, 
	one or more cell (point) locations for which point time series data are written.  There can be
	multiple point TS each for different variables.  Thus, we make a separate file for each variable 
	(unknown until user chooses) that has multiple columns for multiple points (unknown until user chooses). 
	
	\param pSeries An array of SeriesParm structs for point time series output
	\param nSeries Number of point time series occurances
	
	\remarks A structure should/could be set up for this, but to maintain minimal changes between the 
	parallel and serial versions of the code (ELMv1.0), for now we just maintain the original 
	data structure and use some counters below to keep track of diff files and numbers of columns 
	(also allowing different output steps as chosen by the user). (Note: The original SME code 
	printed all variables in one column in one file). 
	
	\note TODO: this whole point-time-series output functionality needs improvement (not high priority, Jan 2005).
	One of the more important capabilities is to have ability to have irregular, calendar-month/year etc output intervals. */

void open_point_lists(SeriesParm *pSeries, int nSeries)
{
  FILE *oFile;
  int i, j, ix, iy, k= 0 ;

  for (i=0; i<nSeries; i++) {
     if( pSeries[i].data == NULL ) {
        if ( numPtFiles>0 ) fclose(oFile); /* in (rare?) case where user asks for no point output, no files stream to close */
        return;
      }
      if (strcmp(pSeries[i].name,pSeries[i-1].name) != 0) {
              /* this is new variable, thus new file.  numPtFiles== increment counter of number of variables (one file per variable) used for output */
          if (i>0) { 
            numPtFiles++; 
            fclose (oFile); 
          }
          
          sprintf(msgStr,"%s%s/Output/PtSer/%s.pts",OutputPath,ProjName,pSeries[i].name);
          if( (oFile = fopen(msgStr,"w") ) == NULL) { 
              fprintf(stderr,"\nERROR, unable to open %s point time series output file.",msgStr); 
              exit(-1); 
          }
          fprintf (oFile, "%s %s %s %s scenario: \nDate\t", &modelName, &modelVers, &SimAlt, &SimModif);
          
      }
/* populate the grid cell locations in the header for this variable/file */
      ix = pSeries[i].Loc.x;
      iy = pSeries[i].Loc.y;
      fprintf(oFile,"%s:(%d,%d)\t",pSeries[i].name,ix,iy);
      cell_pts[numPtFiles]++; /* count the number of cell point locs per variable (1 var per file) */
      
  }
}


/*! \brief Point time series output: print time series data to file(s).  

	\ref open_point_lists Has documentation relevant to this function
	
	\param pSeries A struct of the variables to have point time series output
	\param nSeries Number of point time series occurances
	
 */
void send_point_lists2(SeriesParm *pSeries, int nSeries)
{
  FILE *oFile;
  int i=0, ii, j, ix, iy, k= 0, last_pt;
  int yr_pt[2],mo_pt[2],da_pt[2],hr_pt[2],mi_pt[2];    
  double se_pt[2];           
  double Jdate_pt;
  
  if (numPtFiles==0) return;
  /* k = file/variable counter; i = indiv cellLoc time series counter */
  for (k = 0; k <= numPtFiles; k++) { /* loop over the number of point ts files */
      sprintf(msgStr,"%s%s/Output/PtSer/%s.pts",OutputPath,ProjName,pSeries[i].name);
      if( (oFile = fopen(msgStr,"a") ) == NULL) { 
          fprintf(stderr,"\nERROR, unable to open %s point time series output file.",msgStr); 
          exit(-1); 
      }

      for(j=pSeries[i].laststep; j<pSeries[i].Length; j++ ) { /*temporal loop */
          Jdate_pt = Jdate_init+j*pSeries[i].outstep;
          calcdate( Jdate_pt, mo_pt, da_pt, yr_pt, hr_pt, mi_pt, se_pt); /* get the calendar date info from the julian date */
          fprintf(oFile,"\n%d/%d/%d\t",yr_pt[0],mo_pt[0],da_pt[0] ); /* calendar date in col 1 */

          last_pt = i+cell_pts[k]; /* # of last cellLoc point of the current variable's file */
          for (ii=i; ii<last_pt; ii++) {/* loop over number of point locations per file */
              fprintf(oFile,"%f\t",pSeries[ii].data[j]);
          }
      }
      fclose (oFile); 
      pSeries[i].laststep = j; /* remember the last temporal outstep for this file */
      i += cell_pts[k];     /* increment the indiv cellLoc time series counter */ 
  }
  return;
}


/*! \brief Write to a spatial data window in a debug file.  

	\param fValue Model variable (local) array data
	\param label Output message label (w/ variable name)
	\param desc Message describing the output
	\param N0 Range in x (row!) values
	\param N1 Range in y (col!) values
	\param Mtype General output format type
	\param format Numeric format type
	
 */
void writeSeries(void* fValue, char* label, char* desc, int N0, int N1, byte Mtype, byte format )
{
	/* check on these! (who wrote this? Why? (HCF Dec'04) )*/
  int ix, iy; 
  static unsigned char first_write = 1;
  long int  ret, dimsizes[2];
  unsigned short int refnum;
  if(format == 'H') {
  
#if HDF	  
	  dimsizes[0] = N0;
	  dimsizes[1] = N1;
	  sprintf(cTemp,"%s%s:Output:Windows.hdf",OutputPath,ProjName);
	  DFSDclear();
	  
	  switch(Mtype) {
		 case 'f': 
		    DFSDsetNT(DFNT_FLOAT32);  break;
		 case 'L': 
		    DFSDsetNT(DFNT_FLOAT32);  break;
		 case 'E': 
		    DFSDsetNT(DFNT_FLOAT32);  break;
		 case 'd': case 'i': 
		    DFSDsetNT(DFNT_INT32);  break;
		 case 'c': 
		    DFSDsetNT(DFNT_UINT8);  break;
	  }
	  if(first_write) {
	  	ret = DFSDputdata(cTemp,2,dimsizes,fValue);
		first_write = 0;
	  } else {
	  	ret = DFSDadddata(cTemp,2,dimsizes,fValue);
	  }
	  HDF_VERIFY("DFSDadddata");
	  refnum = DFSDlastref();
	  ret = DFANputlabel(cTemp,DFTAG_SDG,refnum,label);
	  HDF_VERIFY("DFANputlabel");
	  ret = DFANputdesc(cTemp,DFTAG_SDG,refnum,desc,strlen(desc));
	  HDF_VERIFY("DFANputdesc");
#endif
  }
  else {
    fprintf(Driver_outfile,"\n_________%s\n",label);	
    fprintf(Driver_outfile,"%s\n",desc);	
    for (ix=0; ix<N0; ix++) {
      fprintf(Driver_outfile,"\n");
      for (iy=0; iy<N1; iy++) {
        switch(Mtype) {
          case 'f': 
      	 	if(format=='L') 		fprintf(Driver_outfile,"%f ",*(((float*)fValue)+iy+ix*N1)); 
      		else if(format=='E')  	fprintf(Driver_outfile,"%.3E ",*(((float*)fValue)+iy+ix*N1)); 
			else 				fprintf(Driver_outfile,"%.3f ",*(((float*)fValue)+iy+ix*N1));
			break;
      	case 'd': case 'i': 
			fprintf(Driver_outfile,"%d ",*(((int*)fValue)+iy+ix*N1)); 
			break;
      	case 'c': 
			fprintf(Driver_outfile,"%x ",*(((UCHAR*)fValue)+iy+ix*N1)); 
			break;
        }
      }
    }  
  }
}

/*! \brief A variety of stat summaries in a selected spatial window (debug-related).  

	This determines maximum, minimum, sum, average, either cumulatively over iterations or
	not cumulatively.  This Model.outList command has not been used in ELM, and
	has not been verified for accuracy.

	\param fValue Model variable (local) array data
	\param label variable name
	\param nComp number of comps
	\param cType The combination (stats) type
	\param step The current iteration number
	
 */
void Combine(float* fValue, char* label, int nComp, int cType, int step)
{
	int print, cIndex, i, cum;
	static int type = 11;
	char tmpStr[50];
	if( ++type > 99) type = 11;

	cum = ((cType == kAVECUM) || (cType == kSUMCUM)) ? 1 : 0;	
	if(cum) { cIndex = getCombineIndex(label,step,cType,&print); }
	
	switch(cType) {
		case kMAXMIN: 
			sprintf(msgStr,"\nMAXMIN(%d) for %s:",step,label); 
			for(i=0; i<nComp; i++) { sprintf(tmpStr," %f",fValue[i]); strcat(msgStr,tmpStr); }
			WriteMsg(msgStr,1);
		break;
		case kMAX:
			sprintf(msgStr,"\nMAX(%d) for %s:",step,label); 
			for(i=0; i<nComp; i++) { sprintf(tmpStr," %f",fValue[i]); strcat(msgStr,tmpStr); }
			WriteMsg(msgStr,1);
		break;
		case kMIN:
			sprintf(msgStr,"\nMIN(%d) for %s:",step,label); 
			for(i=0; i<nComp; i++) { sprintf(tmpStr," %f",fValue[i]); strcat(msgStr,tmpStr); }
			WriteMsg(msgStr,1);
		break;
		case kSUM:
			sprintf(msgStr,"\nSUM(%d) for %s:",step,label); 
			for(i=0; i<nComp; i++) { sprintf(tmpStr," %f",fValue[i]); strcat(msgStr,tmpStr); }
			WriteMsg(msgStr,1);
		break;
		case kAVE:
			sprintf(msgStr,"\nAVE(%d) for %s:",step,label); 
			for(i=0; i<nComp; i+=2) { sprintf(tmpStr," %f",fValue[i]/fValue[i+1]); strcat(msgStr,tmpStr); }
			WriteMsg(msgStr,1);
		break;
		case kSUMCUM:
			if( print == -1) {
				for(i=0; i<nComp; i++) ctable[cIndex].fvalue[i] = fValue[i]; 
			
			} else {
				for(i=0; i<nComp; i++) ctable[cIndex].fvalue[i] += fValue[i]; 
			}
			if(print==1) { 				
				sprintf(msgStr,"\nSUMCUM(%d) for %s:",step,label); 
				for(i=0; i<nComp; i++) { sprintf(tmpStr," %f",ctable[cIndex].fvalue[i]); strcat(msgStr,tmpStr); }
				WriteMsg(msgStr,1);
			}
		break;
		case kAVECUM:
			if( print == -1) {
				for(i=0; i<nComp; i++) ctable[cIndex].fvalue[i] = fValue[i]; 
			
			} else {
				for(i=0; i<nComp; i++) ctable[cIndex].fvalue[i] += fValue[i]; 
			}
			if(print==1) { 				
				sprintf(msgStr,"\nAVECUM(%d) for %s:",step,label); 
				for(i=0; i<nComp; i+=2) { sprintf(tmpStr," %f",ctable[cIndex].fvalue[i]/ctable[cIndex].fvalue[i+1]); strcat(msgStr,tmpStr); }
				WriteMsg(msgStr,1);
			}
		break;
	}
}	

/*! \brief A variety of stat summaries in a selected spatial window (debug-related).  

	This determines maximum, minimum, sum, average, either cumulatively over iterations or
	not cumulatively.  This Model.outList command has not been used in ELM, and
	has not been verified for accuracy.

	\param name variable name
	\param step The current iteration number
	\param type  The combination (stats) type
	\param last 
	\return index of the count
 */
int getCombineIndex( char* name, int step,int type,int *last)
{
	int count, index = -1;
	*last = 0;
	
	for(count = 0; count < max_combos_open; count++) {
	
		if( ctable[count].free )	{			/* Determine if the slot is in use. */
			if(index == -1) index = count;		/* Nope, keep track of first free global stream table slot. */ 
			if(debug) { fprintf(dFile,"cnt = %d, index = %d, free combo stream\n",count,index); fflush(dFile); }
		}
		else if( strncmp(name,ctable[count].name,25) == 0 && ctable[count].type == type ) {
		    *last = 1; 
		    if(debug) { 
		      fprintf(dFile,"\n(%s)combo cnt = %d, step = %d, type = %d\n",
			      name,count,step,type); 
		      	fflush(dFile); 
		    }
		    return count;
		}
	}
	if( index == -1 ) { 
		index = max_combos_open; 
		if( ++max_combos_open > MAXCOMBOS) { sprintf(msgStr,"Out of combo slots: %s, step=%d, type=%d\n",name,step,type);  usrErr(msgStr); return -1; }
	}
	if(debug) { fprintf(dFile,"\n(%s)index = %d, step=%d, set-up combo\n",name,index,step); fflush(dFile); }
	strncpy(ctable[index].name,name,25);
	ctable[index].step = step;
	ctable[index].type = type;
	ctable[index].free = 0;
	*last = -1;
	return index;
}
/*! \brief Open debug-related output files.

	The Driver0.out and Driver1.out are two output files with debug and/or standard 
	error-warning information.  They are always produced, but have varying
	levels of detail (esp. Driver1.out) depending on the user-selected
	(at run-time) level of the "debug" variable.  Driver0.out basically provides
	initial, static information on data and setup of the model.  Driver1.out 
	provides dynamic information on the simulation.  In sensitivity analysis mode,
	subsequent Driver2.out, Driver3.out, etc provide dynamic information on each 
	new simulation.
	\param index Index to indicate file name w/ "0", "1", and subsequent runs under a sensitivity analysis
*/
void open_debug_outFile(int index)
{
  char filename[120];
  
  sprintf(filename,"%s/%s/Output/Debug/Driver%d.out",OutputPath,ProjName,index);

  Driver_outfile = fopen(filename,"w");
  if(Driver_outfile == NULL) { 
  	fprintf(stderr,"Error, unable to open %s file.\n",filename);
  	fprintf(stderr,"OutputPath: %s\n",OutputPath);  
  	fprintf(stderr,"Project: %s\n",ProjName);  
  	exit(0); 
  }

}

/*! \brief Get format definition of output configuration file (does nothing effective!).

	Identifies the leading characters in the Model.outlist output configuration file that define its format,
	which will never change from it's value of "#2".  (This is another SME component of past flexibility.)
	\param vpFile Pointer to the (model outlist) configuration file
	\param term1 The character "#"
	\param term2 The character "*"
	\param term3 The character "@"
	\param term4 The character "~"
	\return The (integer) format of the configuration file
*/
int init_config_file(FILE *vpFile, char term1, char term2, char term3, char term4)
{
  char test;
  int format,size=-1;

  gTerm[0] = term1;
  gTerm[1] = term2;  
  gTerm[2] = term3;  
  gTerm[3] = term4;  
  skip_white(vpFile);
  test = fgetc(vpFile);
  if(test == gTerm[0])
      fscanf(vpFile,"%d",&format);
  else
      format = -1;
  return format;
}

/*! \brief Skip spaces in reading (model outlist) configuration file 

	\param vpFile Pointer to file being read
	\param tch A space
	\return negative or success
*/
int skip_cnfg_space(FILE *vpFile, char* tch)
{
	char ch; int rv = 1;
	ch = *tch;
	while( isspace(ch) ) { 
		if( (ch=fgetc(vpFile)) == EOF || ch == gTerm[3] || ch == '\0' ) {
			fclose(vpFile); cnfgFile = NULL; return -3; 
		}
	}
	*tch = ch;
	return rv;
}

/*! \brief Parse through a configuration fileline to obtain the configuration commands.

	Populates gCArg[][] with command info.  In very early versions of ELM, this was used in reading the configuration
	file that translated "Stella" model equations into spatial (SME) code.  Thus,
	much of the detail here is unused (ELM v1.0 and later) when reading the
	output (e.g., Model.outList) configuration file (both w/ similar syntax).

	\param vpFile Pointer to the (model outlist) configuration file
	\param nArgs Number of arguments in command
	\param test A character that will be used to indicate an array (in outlist, always is array, char="*")
	\return The index (aka index sequence of variables) of the ViewParm
*/
int parse_packet( FILE *vpFile, int* nArgs, char* test) 
{
    static	int     gVPindex = -1;
    static	char    gCnfg;
    
	char ch = ' ', eChar = ' '; 
	int btype=0, argc=2, go=1, i=0, j=0, new_name = 0;

	if(vpFile == NULL) return -3;
	while (1) {
		if ( skip_cnfg_space(vpFile, &ch) < 0) return -3; j=0;
		if( ch == gTerm[1] || ch == gTerm[2] ) {
		    gCnfg = ch; ch = ' ';
			if ( skip_cnfg_space(vpFile, &ch) < 0 ) return -3;
			while ( !isspace(ch) ) { cTemp[j++] = ch; ch=fgetc(vpFile); }
			cTemp[j++] = '\0'; new_name = 1;
			gVPindex++;
		}
		else break;
	}
	strcpy(gCArg[0],cTemp);
	*test = gCnfg;
	while ( isalnum( ch ) ) { gCArg[1][i++]=ch; ch=fgetc(vpFile); }
	gCArg[1][i]='\0'; i=0;
	while( 1 ) {
		if( ch == '(' ) { eChar = ')'; break;} 
		else if( ch == '[' ) { eChar = ']'; break;} 
		else if( ch == '{' ) { eChar = '}'; break;} 
		else if( !isspace(ch) ) { return -3;}
		ch=fgetc(vpFile);
	}
	while (go) {
		while( ch=fgetc(vpFile) ) {
			if( ch == ',' ) { argc++; break; }
			if( ch == ')' ||  ch == '}' ||  ch == ']')  { 
				if( ch == eChar ) {  argc++; go=0; break;} 
				else { printf( "\nWarning: Syntax error in parse_config, var: %s\n",gCArg[0]); return -2; } 
			}
			gCArg[argc][i++]=ch;
			if( i==(kCArgWidth-1) ) break;
		}
		if(i==0) { printf( "\nWarning: Syntax error in parse_config, var: %s\n",gCArg[0]); return -2; } 
		else { gCArg[argc-1][i]='\0'; i=0; }
		if(argc == kCArgDepth) { 
			while( ch != eChar ) ch=fgetc(vpFile);  
			go = 0; 
		}
	}
	*nArgs = argc;
	if(argc==0) { printf( "\nWarning: Syntax error in parse_config, var: %s\n",gCArg[0]); return -2; }
	else return gVPindex;		
} 


/*! \brief Get a numeric digit from a file.

	\param infile file pointer 
	\return The digit
*/
int get_number( FILE *infile ) {
	char ch;
	int rv;
	
	ch = fgetc(infile);
	if( !isdigit( ch ) ) return(-1);
	rv = ch - '0';
	
	while( isdigit( ch = fgetc(infile) ) ) 
		rv = rv*10 + ( ch - '0' );
		
	return rv;
}

/*! \brief Go to a particular indexed/demarked point in the habitat-specfic parameter file.

	\param infile file pointer 
	\param tchar special character that denotes either habitat or module/sector
	\param index index that points to either habitat or module/sector number 
	\return succes/fail
*/
int goto_index ( FILE *infile, char tchar, int index) 
{
	int rv=1, current_index=-1, itest;

	while(current_index != index) {	
		itest = find_char (infile, tchar);
		if(itest <= 0) return 0;
		current_index = get_number(infile);
		if(current_index < 0) fatal("Bad number format in dBase");
	}
	return 1;
}

/*! \brief Get the N'th parameter in the habitat-specfic parameter file.

	After finding the correct habitat and module/sector location in the file, get
	the value of a parameter.

	\param infile file pointer 
	\param pIndex the index number of a parameter within a module set of parms
	\param end denotes end of file
	\param hIndex the index number of current habitat being read
	\return the parameter value
*/
float get_Nth_parm( FILE *infile, int pIndex, int* end, int hIndex ) 
{
	int i=1, itest=1;
	float rv; char ch;
	
	while ( (i++ < pIndex) && itest > 0 ) { itest = find_char ( infile, '\t' ); }
	if( itest == 0 ) { *end = 1; return(0.0); }
	
	itest = fscanf(infile,"%f",&rv);
	if(itest==1) { 
	if(debug) { 
	   sprintf(msgStr,"Habitat %d:\t%f",hIndex,rv); 
	   WriteMsg(msgStr,1); 
	}
	}
	else if ( (ch = fgetc(infile)) == EOF ) { *end = 1; return(0.0); }
	else {
		sprintf(msgStr,"Read Error in dBase(%d)\n REad Dump:\n",itest); WriteMsg(msgStr,1); usrErr(msgStr);
		for(i=0; i<12; i++) { 
			ch = fgetc(infile);
			if(ch==EOF) { sprintf(msgStr,"\nAt EOF\n"); WriteMsg(msgStr,1); break; }
			else fputc(ch,dFile);
		}
		exit(0);
	}
	*end = 0;
	return(rv);
}


/*! \brief Psuedo-random number generator (unused) */
float SMDRAND(float fminVal, float fmaxVal )
{ 
  float rv, tmp;
  tmp = rand();
  rv = fminVal + (fmaxVal-fminVal)*(tmp/RAND_MAX); 
  return(rv);
} 


/*! \brief Does very little in serial implementation (opens a low-level debug file).

	Primarily intended for parallel apps: \n
	establish processor attributes, open a low-level debug file 
	(in the user's current directory, set by the model execution 
	script ("go"), and seed the pseudo-random number. 
	\param argc number of command line arguments 
	\param argv command line argument
	\return void
*/
void local_setup(int argc, char** argv)
{
  int i;
  char debugfileName[300];
  nprocs[0] = 1;
  nprocs[1] = 1;
  tramType =  0;
  procnum = 1;
  recpnum[0] = 0;
  recpnum[1] = 0;
  Lprocnum = 1;
  /* the combo_table (ctable) used in Combine operation for spatial array summaries (debug-related) */ 
  max_combos_open = 0;	
  for(i = 0; i < MAXCOMBOS; i++) {
    ctable[i].ocnt = 0; 
    ctable[i].free = 1;
  }

  /* this debug output file is not used extensively in current code - it is opened before we get
     environment vars, thus is written to user's current directory ($ModelPath/$ProjName/Load if
     ELM is executed from the "go" script) */ 
 dFile = fopen("ELM.debug","w");
  if(dFile == NULL) { 
    usrErr("Can't open ELM.debug file."); 
    exit(0); 
  } 
  /* there are no random processes in current ELM (v2.4) w/o fire */ 
  srand(seed);
  fprintf(dFile," RAND_MAX = %d\n",(int)RAND_MAX);
}

/*****/
/* begin parallel, effectively unused, code (ex== EXpress parallel compiler compatibility) */
/*****/

/*! \brief Parallel code: effectively unused in serial implementation */
void exparam( struct nodenv* envInfo)
{	
	envInfo->procnum = 1;
	envInfo->nprocs = 1;
	envInfo->groupid = 0;
	envInfo->taskid = 0;
 
}

/*! \brief Parallel code: does nothing in serial implementation) */
int  exgridinit(int dim, int* nprocs) 	
{ return 0;}

/*! \brief Parallel code: effectively unused in serial implementation */
void exgridsplit(int nprocs, int ndim,int nprocs2[2])
{
  nprocs2[0] = 1;
  nprocs2[1] = 1;
}

/*! \brief Parallel code: effectively unused in serial implementation */
void exgridcoord(int pnum, int rnum[2])
{
    tramNum[0] = rnum[0] = 0;
    tramNum[1] = rnum[1] = 0;
}

/*! \brief Parallel code: effectively unused in serial implementation */
void exgridsize( int pnum, int gsize[2],int lsize[2],int lstart[2])
{
	int rem[2], i, j;
	for( i=0; i<2; i++ ) {
		lsize[i] = gsize[i]/nprocs[i];
		rem[i] = gsize[i] - lsize[i]*nprocs[i];
		if(recpnum[i]<rem[i]) lsize[i]++;
		for(j=0; j<recpnum[i]; j++) if( j<rem[i] && recpnum[i] >= rem[i] ) lstart[i] += (lsize[i]+1); else lstart[i] += lsize[i];
	}
}

/*! \brief Parallel code: does nothing in serial implementation) */
void set_async_mode(FILE* file) 
{ return;}

/*! \brief Parallel code: does nothing in serial implementation) */
void fmulti(FILE* file) 
{ return;}

/*! \brief Parallel code: does nothing in serial implementation) */
void fsingl(FILE* file) 
{ return;}

/*! \brief Parallel code: does nothing in serial implementation) */
void fasync(FILE* file) 
{ return;}


/*! \brief Parallel code: does nothing in serial implementation) */
void exchange_borders(UCHAR* map, int size) 
{return;}

/*! \brief Parallel code: does nothing in serial implementation) */
int on_this_proc(int x,int y)
{ return 1; }

/*! \brief Parallel code: does nothing in serial implementation) */
void Cplot(VOIDP Map, unsigned char Mtype, float max_value, float min_value) 
{}

/*! \brief Parallel code: does nothing in serial implementation) */
void broadcastMsg( UCHAR* msgPtr)
{}

/*! \brief Parallel code: does nothing in serial implementation) */
void broadcastInt(int* iValPtr )
{}

/*! \brief Parallel code: does nothing in serial implementation) */
void broadcastChar( UCHAR* cPtr)
{}

/*! \brief Parallel code: does nothing in serial implementation) */
void broadcastData( void* dataPtr, int *dataSize)
{}

/*! \brief Parallel code: does nothing in serial implementation) */
void sync_processors() 
{return;}

/*! \brief Parallel code: does nothing in serial implementation) */
void broadcastFloat(void* dataPtr)
{}

/*****/
/* end parallel, effectively unused, code (ex== EXpress parallel compiler compatibility) */
/*****/

/****************************************************/
