/***************************************************************************
                          gridmap.cpp  -  description
                             -------------------
    begin                : Tue Feb 11 2003
    copyright            : (C) 2003 by Beheen Trimble
    email                : btrimble@sfwmd.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <exception>

#include "consolemsg.h"
#include "keywordmgr.h"
#include "grid.h"
#include "cell.h"
#include "gridmap.h"
#include "util.h"


/* This class finds the common coordinates of the 2 grids
 * cell by cell and assigns the cell number of the source
 * grid to the destination grid. Later on this mapping can
 * be used to resample the data from the source to destination.
 *
 * The GridMap object contains a Console Message (to report
 * messeges, errors, and other events), a Keyword Manager
 * (keyword manipulation routines), bunch of keyword objects,
 * and 2 Grid objects.
 * All the data members are getting created here.
 */
GridMap::GridMap() {

  // added -fcheck-new flag to make new return NULL with new g++   
  console = new ConsoleMsg("GridMap");
  if(console == NULL) {
    cerr << endl;
    cerr << "GridMap" << " ==> " << "!!!!!!!!!!!!!!!!!!!!!!!!!!! =:( !!!! " << endl;
    cerr << "GridMap" << " ==> " << "Creating GridMap Object" << endl;
    cerr << "GridMap" << " ==> " << "!!!!!!!!!!!!!!!!!!!!!!!!!!! =:( !!!! " << endl;
    this->~GridMap();
  }

  keyMgr = new KeywordMgr();
  if(keyMgr == NULL)  {
    this->~GridMap();
  }

  // Initialize the keywords data member
  int status = initKeywords();
  if(status != console->xstatus) {
    this->~GridMap();
  }

  // Init the grids objects,
  // This has to be called after initKeywords()
  DestinationGrid = new Grid("DestGrid");
  if(DestinationGrid  == NULL)  {
    this->~GridMap();
  }
  
  SourceGrid = new Grid("SrcGrid");
  if(SourceGrid == NULL)  {
    this->~GridMap();
  }
}


GridMap::~GridMap() {

  #ifdef DEBUG
  if(console)
    console->MsgDelete("GridMap");
  #endif

  if(console) {
    console->~ConsoleMsg();
    console = NULL;
  }
  if(keyMgr) {
    keyMgr->~KeywordMgr();
    keyMgr = NULL;
  }
  if(DestinationGrid) {
    DestinationGrid->~Grid();
    DestinationGrid = NULL;
  }
  if(SourceGrid) {
    SourceGrid->~Grid();
    SourceGrid = NULL;
  }
  
  // this takes care of all the key words in
  // this source
  map<string,Keyword*>::const_iterator itr;
  if(keywords.size() >  0) {
    for(itr=keywords.begin(); itr!=keywords.end(); ++itr) {
      Keyword* k = itr->second;
      if(k) {
        k->~Keyword();
        k=NULL;
      }
    }
  }
}

/*
 * This method deligates reading the configuration file
 * to keyword manager, since manager knows about keyword
 * objects. The result are a bunch of keyword objects that
 * they can be used by other classes.
 */
int GridMap::readConfigFile(string file) {

  // Initialize 
  int status = console->xstatus;

  status = keyMgr->readCfgFile(file);
  if (status != console->xstatus) {
    console->ErrMsg("GridMap","readCfgFile");
    return status;
  }

  string keyStart = "DESTINATION";
  string keyEnd =   "END_DEST";
  
  // Keyword mgr checks the format of the user's config file
  // Only the start and end keywords are checked!
  status = keyMgr->checkFormat(keyStart,keyEnd);
  if(status != console->xstatus) {
    console->ErrMsg("GridMap","File Format");
    return status;
  }

  // Keyword mgr parses different blocks of config file
  // See the class diagram for relationship among the objects.
  status = keyMgr->parseInputBlock(keywords,keyStart,keyEnd);
  if(status != console->xstatus) {
    console->MsgFail("GridMap","parseInputBlock");
    return status;
  }

  // Now repeat the same process for the Source block
  keyStart = "SOURCE";
  keyEnd = "END_SRC";

  status = keyMgr->checkFormat(keyStart,keyEnd);
  if(status != console->xstatus) {
    console->ErrMsg("GridMap","File Format");
    return status;
  }

  status = keyMgr->parseInputBlock(keywords,keyStart,keyEnd);
  if(status != console->xstatus) {
    console->MsgFail("GridMap","parseInputBlock");
    return status;
  }  
  
  #ifdef DEBUG
  console->DebugMsg("GridMap"," "," ");
  console->PrintKeywordMap(&keywords);
  console->DebugMsg("GridMap"," "," ");
  keyMgr->printInputBlock();
  #endif

  return status;
}


/* This method creates a list of keywords
 * that are required for GridMap to be able
 * to create the 2 grids to map. The keyword
 * objects contains following members:
 * keyword, type, meaning, and value fields.
 * All the fields are populated and initialized
 * here. This list is used only
 * as a template to compare with the keywords
 * read from the user configuration file. Any new
 * keyword must be added to this method and
 * must be updated in readGridFile method,
 * populateSourceGridFromBinaryFile method,
 * and populateGrid method. I hope I did not
 * miss any!
 * Note: key start and key ends have to be
 * updated in readGridFile method.
 */
int GridMap::initKeywords(void) {

  int status = console->xstatus;

  keyList.push_back("DESTINATION");
  keyList.push_back("END_DEST");
  keyList.push_back("SOURCE");
  keyList.push_back("END_SRC");
  keyList.push_back("X_ORIGIN");
  keyList.push_back("Y_ORIGIN");
  keyList.push_back("X_DIRECTION");
  keyList.push_back("Y_DIRECTION");
  keyList.push_back("NUM_ROW");
  keyList.push_back("NUM_COL");
  keyList.push_back("TIMESTEP");
  keyList.push_back("NUMBERING");
  ////////////////////////////////////
  Keyword *key = new Keyword();
  
  key->keyword = keyList[0];    //"DESTINATION";
  key->meaning = "Gets the cell values of the source grid ";
  key->type = "STRING";
  key->svalue = "Destination grid name";
  keywords[key->keyword] = key;

  key = new Keyword();       //"END_DEST";
  key->keyword = keyList[1]; // this can not be "END_DESTINATION!
                             // give problem in checkFormat because
                             // of using fileLine.find(...).
                             // DESTINATION and END_DESTINATION
                             // is same to find!
  key->meaning = "End Destination";
  key->type = "STRING";
  key->svalue = "Marks the end of the Destination block";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[2];    //"SOURCE";
  key->meaning = "Cell's value of this grid to be used by Destination grid";
  key->type = "STRING";
  key->svalue = "Source grid name";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[3];    //"END_SRC";
  key->meaning = "End Source";
  key->type = "STRING";
  key->svalue = "Marks the end of the Source block";
  keywords[key->keyword] = key;
  
  // TERM keywords
  key = new Keyword();
  key->keyword = keyList[4];    //"X_ORIGIN";
  key->meaning = "X coordinate of the grid origin";
  key->type = "DOUBLE";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[5];    //"Y_ORIGIN";
  key->meaning = "Y coordinate of the grid origin";
  key->type = "DOUBLE";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[6];    //"X_DIRECTION";
  key->meaning = "The grid can be constructed to the left or to the right direction";
  key->type = "STRING";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[7];    //"Y_DIRECTION";
  key->meaning = "The grid can be constructed upward or downward";
  key->type = "STRING";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[8];    //"NUM_ROW";
  key->meaning = "Number of grid rows";
  key->type = "INT";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[9];    //"NUM_COL";
  key->meaning = "Number of grid columns";
  key->type = "INT";
  keywords[key->keyword] = key;

  key = new Keyword();
  key->keyword = keyList[10];    //"TIMESTEP";
  key->meaning = "Grid cell size in x and y direction. Only square grid is implemented.";
  key->type = "FLOAT";
  keywords[key->keyword] = key;


  key = new Keyword();
  key->keyword = keyList[11];    // NUMBERING
  key->meaning = "Grid row and column starts at 0,0 or 1,1";
  key->type = "INT";
  keywords[key->keyword] = key;
  
  return status;
}


/* All the grid information are already saved in
 * the form of keywords. This method uses the
 * populated keywords and transforms them into
 * Grid objects. 
 */
int GridMap::populateGrids(void) {

  int status = console->xstatus;
  
  string dKey = keyList[0];   // DESTINATION
  string sKey = keyList[2];   // SOURCE
  
  // destKey contains the entire destination block terms
  map<string,Keyword*> destKey = keyMgr->getInputBlock(dKey);
  map<string,Keyword*> srcKey = keyMgr->getInputBlock(sKey);

  if(destKey[dKey] == NULL || srcKey[sKey] == NULL) {
     status = console->estatus;
     console->MsgFail("GridMap","getInputBlock");
     return status;
  }
  else { // assign the data members of the grid
    Keyword* dky = NULL;
    Keyword* sky = NULL;
    
   
    for(unsigned int i=0; i<keyList.size(); i++) {
      switch(i) {
        case 0 :
          dky = destKey[keyList[i]];
          DestinationGrid->setName(dky->svalue);
          break;
        case 1 :
          // do nothing
          break;
        case 2 :
          sky = srcKey[keyList[i]];
          SourceGrid->setName(sky->svalue);
          break;
        case 3 :
          // do nothing
          break;
        case 4 :
          dky = destKey[keyList[i]];  
          sky = srcKey[keyList[i]];  
          DestinationGrid->setX_Origin(dky->dvalue);          
          SourceGrid->setX_Origin(sky->dvalue);
          break;
        case 5 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setY_Origin(dky->dvalue);
          SourceGrid->setY_Origin(sky->dvalue);
          break;
        case 6 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setX_Direction(dky->svalue);
          SourceGrid->setX_Direction(sky->svalue);
          break;
        case 7 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setY_Direction(dky->svalue);
          SourceGrid->setY_Direction(sky->svalue); 
          break;
 
        case 8 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setNumRow(dky->ivalue);
          SourceGrid->setNumRow(sky->ivalue);
          break;
        case 9 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setNumCol(dky->ivalue);
          SourceGrid->setNumCol(sky->ivalue);
          break;
        case 10 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]]; 
          DestinationGrid->setTimeStep(dky->fvalue);
          SourceGrid->setTimeStep(sky->fvalue);
          break;
 
        case 11 :
          dky = destKey[keyList[i]];
          sky = srcKey[keyList[i]];
          DestinationGrid->setNumbering(dky->ivalue);
          SourceGrid->setNumbering(sky->ivalue);
          break;
        default :
          status = console->estatus;
          console->ErrMsg("GridMap","Populating Grids");
          return status;
      } // end of switch
    } // end of for
  } // end of else
 
  return status;
}


/* This method construct the 2 grids from the given information
 * in the config file. A grid is constructed upon construction
 * of its cells. This is for grids in any direction
 */

int GridMap::constructGrids() {

  // This is tested only for source grid cell size > destination
  // grid cell size. Must be tested  for reverse.
  if(SourceGrid->getTimeStep() < DestinationGrid->getTimeStep()) {
    console->ErrMsg("GridMap", "Sorry! This program is only tested for a source grid cell size larger than destination grid cell size");
    console->ErrMsg("GridMap", "Please contact Dr. Carl Fitz at cfitz@sfwmd.gov or switch the grids");
    return console->estatus;
  }

  
  int status = console->xstatus;
  #ifdef DEBUG
  console->DebugMsg("GridMap","Constructing Grids for Source ",SourceGrid->getGridName());
  #endif
  
  SourceGrid->setCorners();
  SourceGrid->printCorners();
  status = SourceGrid->defineCells();
  if(status == console->estatus) {
    status = console->estatus;
    console->MsgFail("GridMap","Constructing Grids for Source");
    return status;
  }

  #ifdef DEBUG
  console->DebugMsg("GridMap","Constructing Grids for Destination",DestinationGrid->getGridName());
  #endif
  
  DestinationGrid->setCorners();
  DestinationGrid->printCorners();
  status = DestinationGrid->defineCells();
  if(status == console->estatus) {
    status = console->estatus;
    console->MsgFail("GridMap","defineCell for Destination Grid");
    return status;
  }
  
  return status;
}



/* This method maps cell numbers of the source grid to
 * cell numbers of the destination grid by coordinates.
 * To save the comparison time, only the common area
 * between the 2 grids are compared. The boundary cells
 * are considered.
 */
int GridMap::mapGrids() {

  int status = console->xstatus;

  // find the search space where the 2 polygons clip, to save search time
  status = defineSearchSpace();
  if(status == console->estatus) {
    console->MsgFail("GridMap","defineSearchSpace");
    return status;
  }
  

  defineEastWestBoundaryCells();
  #ifdef DEBUG
  console->DebugMsg("GridMap"," "," ");
  status = printEastWestBoundary();
  if(status==console->estatus) {
    console->MsgFail("GridMap","Printing East West Boundary");
    return status;
  }
  #endif

  defineSouthNorthBoundaryCells(); 
  #ifdef DEBUG
  console->DebugMsg("GridMap"," "," ");
  status = printSouthNorthBoundary();
  if(status==console->estatus) {
    console->MsgFail("GridMap","Printing South North Boundary");
    return status;
  }
  #endif

  // now save only the unique clipped cells
  // I think this is for use in mapping grids to each other!
  srcClippedGridID();
  destClippedGridID();
  #ifdef DEBUG
  console->DebugMsg("GridMap"," "," ");
  status = printClippedCells();
  if(status==console->estatus)
    return status;
  #endif


  #ifdef DEBUG
  console->DebugMsg("GridMap" ,"Mapping Source Grid to Destination Grid"," ");
  #endif

  // This is tested only for source grid cell size > destination
  // grid cell size. Must be tested  for reverse.

  string srcx_dir = SourceGrid->getX_Direction();
  string srcy_dir = SourceGrid->getY_Direction();
  string destx_dir = DestinationGrid->getX_Direction();
  string desty_dir = DestinationGrid->getY_Direction();

  // clipID variable contains lower left(index 0, index 1), upper right(2,3), lower right(4,5),
  // and upper left(6,7) cell numbers of the src and destination grids
  if(srcx_dir == "RIGHT" && srcy_dir == "UP") {
    if(destx_dir == "RIGHT" && desty_dir == "UP") {
      status = source2DestinationRURU();
      if(status == console->estatus)
        return(status);
    }
    else if (destx_dir == "RIGHT" && desty_dir == "DOWN") {
      status = source2DestinationRURD();
      if(status == console->estatus) 
        return(status);
    }
  }
  
  return status;
}



/* At this time corners of each grid is defined.
 * This method finds the 4 corner coords of the common grid
 * (clip area) and finds the corresponding cell numbers of
 * the 2 grids. The found cell numbers are used in the loop
 * to prevent extra looping through the entire cell numbers.
 * The clip area cell numbers are calculated from lower
 * left upward and from upper right downward.
 * It also creates a map of the east/west boundary cells of
 * both grids and creates vector of all the cells in?? gridmap??
 * This consideres all the directional cases.
 */
int GridMap::defineSearchSpace() {

  #ifdef DEBUG
  console->DebugMsg("GridMap","Defining Search Space"," ");
  #endif
  
  int status = console->xstatus;
  
  Point* sll = SourceGrid->getLowerLeftCorner();
  Point* dll = DestinationGrid->getLowerLeftCorner();
  // this is not a point member, don't use print on point
  Point clipLL = largerXlargerYPoint(sll,dll);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Found Lower Left Corner of Clip Poly"," ");
  console->DebugMsg("GridMap",clipLL.getX(),clipLL.getY());
  #endif
  
  Point* p = new Point(clipLL);
  clipCoord.push_back(p);

  Point* sur = SourceGrid->getUpperRightCorner();
  Point* dur = DestinationGrid->getUpperRightCorner();
  Point clipUR = smallerXsmallerYPoint(sur,dur);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Found Upper Right Corner of Clip Poly"," ");
  console->DebugMsg("GridMap",clipUR.getX(),clipUR.getY());
  #endif
  
  p = new Point(clipUR);
  clipCoord.push_back(p);

  Point* slr = SourceGrid->getLowerRightCorner();
  Point* dlr = DestinationGrid->getLowerRightCorner();
  Point clipLR = smallerXLargerYPoint(slr,dlr);   // smallest x, larger y

  #ifdef DEBUG
  console->DebugMsg("GridMap","Found Lower Right Corner of Clip Poly"," ");
  console->DebugMsg("GridMap",clipLR.getX(),clipLR.getY());
  #endif
  
  p = new Point(clipLR);
  clipCoord.push_back(p);


  Point* sul = SourceGrid->getUpperLeftCorner();
  Point* dul = DestinationGrid->getUpperLeftCorner();
  Point clipUL = largerXsmallerYPoint(sul,dul);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Found Upper Left Corner of Clip Poly"," ");
  console->DebugMsg("GridMap",clipUL.getX(),clipUL.getY());
  #endif
  
  p = new Point(clipUL);
  clipCoord.push_back(p);

  // For West Boundary Cells
  int ScellNo_ll = SourceGrid->findCellNumber(clipLL);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Source Lower Left Grid Cell ID",ScellNo_ll);
  #endif
  
  clipID.push_back(ScellNo_ll);  //0

  int DcellNo_ll = DestinationGrid->findCellNumber(clipLL);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Destination Lower Left Grid Cell ID",DcellNo_ll);
  #endif
  
  clipID.push_back(DcellNo_ll); //1


  // For South Boundary Cells
  int ScellNo_ur = SourceGrid->findCellNumber(clipUR);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Source Upper Right Grid Cell ID",ScellNo_ur);
  #endif
  
  clipID.push_back(ScellNo_ur);   //2

  int DcellNo_ur = DestinationGrid->findCellNumber(clipUR);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Destination Upper Right Grid Cell ID",DcellNo_ur);
  #endif
  
  clipID.push_back(DcellNo_ur);  //3


  // For East Boundary Cells

  int ScellNo_lr = SourceGrid->findCellNumber(clipLR);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Source Lower Right Grid Cell ID ",ScellNo_lr);
  #endif
  
  clipID.push_back(ScellNo_lr);  //4


  int DcellNo_lr = DestinationGrid->findCellNumber(clipLR);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Destination Lower Right Grid Cell ID",DcellNo_lr);
  #endif
  
  clipID.push_back(DcellNo_lr);  //5


 // For North Boundary Cells
  int ScellNo_ul = SourceGrid->findCellNumber(clipUL);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Source Upper Left Grid Cell ID",ScellNo_ul);
  #endif
  
  clipID.push_back(ScellNo_ul);  //6

  int DcellNo_ul = DestinationGrid->findCellNumber(clipUL);

  #ifdef DEBUG
  console->DebugMsg("GridMap","Clipped Destination Upper Left Grid Cell ID ",DcellNo_ul);
  #endif
  
  clipID.push_back(DcellNo_ul);  //7

  delete p;

  return status;
}


int GridMap::defineSouthNorthBoundaryCells() {

  int scell_sid = clipID[0];  // source and dest. clipped cell numbers
  int dcell_sid = clipID[1];  // on lower left corner for south boundary

  int scell_nid = clipID[6]; // on upper left for north boundary
  int dcell_nid = clipID[7];

  float destSz = DestinationGrid->getTimeStep();
  float halfSz = destSz / 2;
  float srcSz = SourceGrid->getTimeStep();


  Point* p1 = clipCoord[0];  // ll
  double xll = p1->getX();

  // with cols, we have to calc which col out first point is!
  Point* p = clipCoord[1];   // upper right clipped coordinates, used
  double xur = p->getX();    // to stop the search loop
 
  /* must adjust to the clip boundary, otherwise the while loop
     below goes more than needed.
   */
  double uloop = destSz;  // init value
  double destsouthx = DestinationGrid->getCellX(dcell_sid);
  // regardless of the numbering starting direction, the largerer x
  // is on east side.
  if(xll >= destsouthx) {
    double diff = xll - destsouthx;
    uloop += (xur - diff);
  }

  double srcsouthx = SourceGrid->getCellX(scell_sid) + srcSz;
  destsouthx += destSz;
 
  while(destsouthx < uloop) {    // comparing lr of each grid cell
    if(destsouthx > srcsouthx) {
      // by how much?
      if(destsouthx - srcsouthx > halfSz) {
        // go to next src
        if(SourceGrid->getX_Direction()=="RIGHT") {  // y dir does not matter
          scell_sid += 1;
          scell_nid += 1;
          srcsouthx += srcSz;
        }
        else {
          cout << "GridMap ==> has not been implemented" << endl;
          this->~GridMap();
          exit(0);
        }
      }
    }
      
    if(DestinationGrid->getX_Direction()=="RIGHT") {  // y dir does not matter
      // recalc new destx
      //cout << destsouthx << " " << dcell_nid << " " << srcsouthx << " " << scell_nid << endl;
      southBoundary[dcell_sid] = scell_sid;
      northBoundary[dcell_nid] = scell_nid;
      dcell_sid += 1;
      dcell_nid += 1;
      destsouthx += destSz;
    }
  }
  //testing
  #ifdef DEBUG
  map<int,int>::const_iterator itr;
  for(itr=northBoundary.begin(); itr!=northBoundary.end(); ++itr) {
    cout << itr->first << " " << itr->second << endl;
  }
  for(itr=southBoundary.begin(); itr!=southBoundary.end(); ++itr) {
    cout << itr->first << " " << itr->second << endl;
  }
  #endif

  return console->xstatus;
 
}



/* To further reduce the comparison between cells, and for other
 * use the boundary cell numbers of the grids are defined here,
 * direction is considered. The results are saved in westBoundary
 * and eastBoundary map and can be printed by printEastWestBoundary()
 * method. By default, when DEBUG is set in make file, the maps are
 * being printed.
 */
int GridMap::defineEastWestBoundaryCells() {

  int status = console->xstatus;

  int ScellNo_ll = clipID[0];  // source and dest. clipped cell numbers
  int DcellNo_ll = clipID[1];  // on lower left corner, for west boundary

  int ScellNo_lr = clipID[4];  // source and dest. clipped cell numbers
  int DcellNo_lr = clipID[5];  // on lower right corner, for east boundary

  Point* p = clipCoord[1];   // upper right clipped coordinates, used
  double yur = p->getY();    // to stop the search loop
  Point* p1 = clipCoord[0];  // ll
  double yll = p1->getY();

  // westBoundary member is a map of dest cell numbers to src
  // cell numbers on the left side of the clipped grids

  int destwestcell = DcellNo_ll;
  int srcwestcell = ScellNo_ll;

  int srceastcell =  ScellNo_lr;
  int desteastcell = DcellNo_lr;

  int destcol = DestinationGrid->getNumCol();
  int srccol = SourceGrid->getNumCol();

  float destwestySz = DestinationGrid->getTimeStep();
  float srcwestySz = SourceGrid->getTimeStep();


  /* must adjust to the clip boundary, otherwise the while loop
     below goes more than needed.Only if ll is > test is needed
   */
  double uloop;
  double destwesty = DestinationGrid->getCellY(DcellNo_ll);
  // regardless of the numbering starting direction, the higher y
  // is on north side.
  if(yll >= destwesty) {
    double diff = yll - destwesty;
    uloop = yur - diff;
  }

  double srcwesty = SourceGrid->getCellY(ScellNo_ll) + srcwestySz;
  // cout << destwesty << " " << srcwesty << " " << SourceGrid->getCellY(ScellNo_ll) << endl;


  while(destwesty < uloop) {  // check for upper right clip boundary

    if(destwesty > srcwesty) {
      // go to next src
      if(SourceGrid->getX_Direction()=="RIGHT" && SourceGrid->getY_Direction()=="UP") {
        srcwestcell += srccol;
        srcwesty += srcwestySz;
        srceastcell += srccol;
      }
      else if(SourceGrid->getX_Direction()=="RIGHT" && SourceGrid->getY_Direction()=="DOWN") {
        srcwestcell -= srccol;
        srcwesty += srcwestySz;  //always upward
        srceastcell -= srccol;
      }
    }


    /* if dest y <= src next y, see how much is smaller
     * if diff is > half of the dest cell then go to
     * one src cell up.
     * example: cell 2161 (elm) vs. cell 125 (nsm/wmm)
     */
    double diff = srcwesty - destwesty;
    if(diff > (destwestySz / 2)) {
      westBoundary[destwestcell] = srcwestcell;
      eastBoundary[desteastcell] = srceastcell;
    }
    else {
      // belong to next src
      if(SourceGrid->getX_Direction()=="RIGHT" && SourceGrid->getY_Direction()=="UP") {
        srcwestcell += srccol;
        srceastcell += srccol;
      }
      else if(SourceGrid->getX_Direction()=="RIGHT" && SourceGrid->getY_Direction()=="DOWN") {
        srcwestcell -= srccol;
        srceastcell -= srccol;
      }

      srcwesty += srcwestySz;  //for all
      westBoundary[destwestcell] = srcwestcell;
      eastBoundary[desteastcell] = srceastcell;
    }

    // increment
    if(DestinationGrid->getX_Direction()=="RIGHT" && DestinationGrid->getY_Direction()=="UP") {
      destwesty += destwestySz ;
      destwestcell += destcol;
      desteastcell += destcol;
    }

    else if(DestinationGrid->getX_Direction()=="RIGHT" && DestinationGrid->getY_Direction()=="DOWN") {
      destwesty += destwestySz ;
      destwestcell -= destcol ;
      desteastcell -= destcol;
    }
  } // end of while y

 return status;
}




/* This method finds all the dest ids that are within
 * the boundary of the src grid. The source and destination
 * grids must exist so does the west/eastBoundary maps before
 * calling this method. The larger cell has duplicates, using
 * map instead to get the unique cell id.
 */
void GridMap::destClippedGridID() {

  int ddist = abs(clipID[1] - clipID[5]);    // number of cols between left & right

  map<int,int>::const_iterator itr;
  map<int,int> lazy_map;
  int k = 0; // dummy
  for(itr=westBoundary.begin(); itr!=westBoundary.end(); ++itr) {
    int did = itr->first;
    for(int i=0; i<= ddist; i++)
      lazy_map[did++] = k++;
  }


  // now put them in a vector
  map<int,int>::const_iterator mitr;
  for(mitr=lazy_map.begin();  mitr!=lazy_map.end(); mitr++) {
    destClippedIDs.push_back(mitr->first);
  }
}



/* This method finds all the src ids that are within
 * the boundary of the dest grid. The source and destination
 * grids must exist so does the west/eastBoundary maps before
 * calling this method. The larger cell has duplicates, using
 * map instead to get the unique cell id.
 */
void GridMap::srcClippedGridID() {

  int sdist = abs(clipID[0] - clipID[4]);    // number of cols between left & right

  map<int,int>::const_iterator itr;
  map<int,int> lazy_map;
  int k = 0; // dummy
  for(itr=westBoundary.begin(); itr!=westBoundary.end(); ++itr) {
    int sid = itr->second;
    for(int i=0; i<= sdist; i++)
      lazy_map[sid++] = k++;
  }


  // now put them in a vector
  map<int,int>::const_iterator mitr;
  for(mitr=lazy_map.begin();  mitr!=lazy_map.end(); mitr++) {
    srcClippedIDs.push_back(mitr->first);
  }
  
}



/* This method maps the cell ids of source grid cells to those
 * cells of the destination grid cells if more than half of the
 * destination cells are inside source cells.
 * This is used when direction of the source grid is from left
 * to right in x direction and upward in y direction, and the
 * destination grid is left to right in x direction and downward
 * in y direction. It accumodates for the different numbering.
 * NOTE: This method is used when src grid cell size is
 * larger than destination grid cell size. For other way around
 * this must be tested!
 */

int GridMap::source2DestinationRURD() {  //source(x->right,y->up), dest(right,down)

  int status = console->xstatus;

  float deststep = DestinationGrid->getTimeStep();
  float midX = deststep / 2;
  int numbering = DestinationGrid->getNumbering();
  int destnumcols = DestinationGrid->getNumCol();
  
  // works only if src time steps are larger than dest time steps
  int ratio =(int)(ceil(SourceGrid->getTimeStep() / deststep));

  // this the Dest grid, not a dest cell.
  // used in finding a dest cell row and col that corresponds
  // to the current src cell. Used upper left because the
  // direction in y is downward and row cols starts from top
  Point* Destpnt = DestinationGrid->getUpperLeftCorner();
  double DestulX = Destpnt->getX();
  double DestulY = Destpnt->getY();
  int dest_start_id = clipID[7];  // dest ul for first cell check
  int dest_end_id = clipID[5];    // dest lr for last cell check


  // the srcClippedIDs vector only contains all the src cells
  // within the boundary, therefore no need to check on
  // out of boundary cells for the src.
  for(unsigned int i=0; i<srcClippedIDs.size(); i++) {
  
    int src_id = srcClippedIDs[i];
      
    Cell* src_cell = SourceGrid->findCellByID(src_id);
    
    Point pur = src_cell->getUpperRight();
    Point pll = src_cell->getLowerLeft();

    double srcurX = pur.getX();
    double srcllX = pll.getX();
    double srcllY = pll.getY();
    double srcurY = pur.getY();


    // at each src cell calculates the row col of the upper right dest cell
    // and calculates the cells to the left and downward to cover all the
    // possible dest cells that falls within this src cell.
    
    // Dest. grid upper left x is always <= srcurx of a src cell if within its boundary.
    int destcol = (int) ceil((srcurX - DestulX ) / deststep) ;  
    // Dest. grid upper left y is always >= srclly of a src cell.                                                            
    int destrow = (int) ceil((DestulY - srcllY) / deststep);    
    
    int destcell_id = (destcol + numbering - 1) + ((destrow + numbering - 1) * destnumcols)  ;

    int ycid = destcell_id, xcid = destcell_id;

    vector<int> cidList;
    for(int i=0; i< ratio; ++i) {  // dest ids inc. going right
      cidList.push_back(xcid);
      xcid--;
    }

    
    for(int j=0; j < (ratio * ratio) ; ++j) {
      ycid = cidList[j] - destnumcols;  // dest ids decr. going up
      cidList.push_back(ycid);
    }

    for(unsigned int k=0; k< cidList.size();++k) {

      // when des id gets to the first cell or last cell goes out of bound.
      int id = cidList[k];
      if(id < dest_start_id || id > dest_end_id)
        continue;
        
      Cell* dest_cell = DestinationGrid->findCellByID(id);

   
      // if src cell is larger, any dest cell between
      // lower left corner and upper right corner of
      // src grid belongs to this src cell.

      Point dpur = dest_cell->getUpperRight();
      double desturX = dpur.getX();
      double destX = desturX - midX;
      double desturY = dpur.getY();
      double destY = desturY - midX; //same as midY


      if(srcllX <= destX && destX <= srcurX && srcllY <= destY && destY <= srcurY)  {
        //this is the destination cell id to src cell ids map or
        //can say is the smaller cell size grid to larger cell size grid.
        one2manyMap[id] = src_id;
        dest_cell->setMappedID(src_id);
      } //end of if

    } // end of for
  }
  return status;
}


/* At the current design, this is very hard to mix the
 * directionality into one method.
 * This method maps the grids when direction on both
 * grids are the same.  This must be changed to something
 * like RURD. This has not been tested!
 */
int GridMap::source2DestinationRURU() {

  int status = console->estatus;
  
  // Main loop for finding: if the center point of the
  // each cell in the src grid belong to which cell in
  // the destination grid. Note that the 'this' pointer
  // is the src grid object.
  // This method is used when src cell is larger than
  // destination cell. 

  int llsrc_id = clipID[0];
  int lldest_id = clipID[1];
  int ursrc_id = clipID[2];
  int urdest_id = clipID[3];
  int i = lldest_id;
  int j = llsrc_id;      // counters

  // get all the cells in the destination grid
  // for the comparison
  vector<Cell*>& destList = DestinationGrid->getCellList();
  float midX = DestinationGrid->getTimeStep() / 2;
  int numbering = DestinationGrid->getNumbering();

  for(; j<=ursrc_id; ) {  // search space cell numbers
  
    Cell* srcCell = SourceGrid->findCellByID(j);
  
    int srcid = srcCell->getID();  // is not same as j all the time

    Point pur = srcCell->getUpperRight();
    Point pll = srcCell->getLowerLeft();
    double srcurX = pur.getX();
    double srcllX = pll.getX();
    double srcurY = pur.getY();
    double srcllY = pll.getY();

    j++;
    
    // i is global
    // this loop maps the cells right ward
    while(i<=urdest_id ) {
    
      Cell* destCell = destList[i-numbering];    // destList[i] !=  destList ID
     
      int destid = destCell->getID();           // they differ by numbering
   
      // if src cell is larger, any dest cell between
      // lower left corner and upper right corner of
      // src grid belongs to this src cell.

      Point dpur = destCell->getUpperRight();
      double desturX = dpur.getX();
      double destX = desturX - midX;
      double desturY = dpur.getY();
      double destY = desturY - midX; //same as midY
    
      if(srcllX <= destX && destX <= srcurX ) {
        if(srcllY <= destY && destY <= srcurY)  {
          one2manyMap[destid] = srcid;
          destCell->setMappedID(srcid);
          ++i;
        }
        else {
          // if between src x but not between src y
          // go to next src.
          break; // src id is already incremented
        }
      } // end of if


      else {  // is a boundary cell and add them to the list too

        int sidw = this->findWestBoundaryCells(destid);  //don't use i and j for
        int side = this->findEastBoundaryCells(destid);  //out of bound cells

        if(sidw != console->estatus ) {
          one2manyMap[destid] = sidw;
          destCell->setMappedID(sidw);

          // as long as next dest cell to process, maps
          // to starting src cell (j), don't increment
          // the src loop.
          j = sidw;
          ++i;
          break;
        }
        else if(side != console->estatus) {
          one2manyMap[destid] = side;
          destCell->setMappedID(side);          
          j = side;
          ++i;
          break;
        }
        else {
          // don't inclrement i
          break;  // goto next src loop
        } // end of else
      } // end of else
    } // end of while destll
  } // end of src
  return status;
}




// given the key (destid), finds the value (srcid)
int GridMap::findWestBoundaryCells(int destid) {

  map<int,int>::const_iterator itr;
  for(itr=westBoundary.begin(); itr!=westBoundary.end(); ++itr) {
    if(itr->first == destid)
      return itr->second;
  }
  return console->estatus;
}


int GridMap::findEastBoundaryCells(int destid) {

  map<int,int>::const_iterator itr;
  for(itr=eastBoundary.begin(); itr!=eastBoundary.end(); ++itr) {
    if(itr->first == destid)
      return itr->second;
  }
  return console->estatus;
}

                                                                    
void GridMap::printOne2manyMap() {

  console->MsgPrint("GridMap", "one2manyMap");
  
  map<int,int>::const_iterator itr;
  cout << "Dest Cell#" << '\t' << "Src Cell#" << endl;
  cout << "==========" << '\t' << "=========" << endl;
  for(itr=one2manyMap.begin(); itr!=one2manyMap.end(); ++itr)
    cout << itr->first << '\t' << '\t' << itr->second << endl;
}


int GridMap::writeBoundaryGrid(string outfile)  {


  int status = console->xstatus;    // Error code variable
  console->MsgWrite("GridMap", "Boundary Grid",outfile);

  ofstream out;

  // Open the output file for writting
  out.open(outfile.c_str(), ios::out);
  if (!out) {
    status = console->estatus;
    console->MsgFail("GridMap","Open File");
    return status;
  }

  out << "This is boundary grid ids of finer grid mapped to coarser boundary grid ids" << endl;
  out << "Note that the mapped id = -1 means no cell intersect between 2 grids." << endl;
  out << "ID        ROW        COL       MAPPED_ID" << endl;
  out << "=========================================" << endl;

  out << "SOUTH BOUNDARY: " << endl;
  map<int,int>::const_iterator sitr;
  for(sitr=southBoundary.begin(); sitr!=southBoundary.end(); ++sitr)
    out << sitr->first << '\t' << '\t' << sitr->second << endl;
  out << "NORTH BOUNDARY: " << endl;
  map<int,int>::const_iterator itr;
  for(itr=northBoundary.begin(); itr!=northBoundary.end(); ++itr)
    out << itr->first << '\t' << '\t' << itr->second << endl;
  out << "EAST BOUNDARY: " << endl;
  for(itr=eastBoundary.begin(); itr!=eastBoundary.end(); ++itr)
    out << itr->first << '\t' << '\t' << itr->second << endl;
  out << "WEST BOUNDARY: " << endl;
  for(itr=westBoundary.begin(); itr!=westBoundary.end(); ++itr)
    out << itr->first << '\t' << '\t' << itr->second << endl;
  out.close();
}



void GridMap::printOne2OneMap()  {

  console->MsgPrint("GridMap", "one2oneMap");
  map<int,int>::const_iterator itr;
  cout << "Dest Cell#" << '\t' << "Src Cell#" << endl;
  cout << "==========" << '\t' << "=========" << endl;
  for(itr=one2oneMap.begin(); itr!=one2oneMap.end(); ++itr) 
    cout << itr->first << '\t' << '\t' << itr->second << endl;
}


void GridMap::printMgrData() {
   keyMgr->printInputBlock();
}


int GridMap:: printSouthNorthBoundary() {

  int status = console->xstatus;
  console->MsgPrint("GridMap", "South North Boundary");
  if(southBoundary.size() == 0) {
    status = console->estatus;
    console->ErrMsg("GridMap","Empty South Boundary Map");
    console->ErrMsg("GridMap","Perhapse, you are calling this method before calling defineEastWestBoundaryCells method");
    return status;
  }


  map<int,int>::const_iterator south;
  map<int,int>::const_iterator north;
 
  cout << "Src South#" << '\t' << "Dest South#" << '\t' << "Src North#" << '\t' << "Dest North#" << endl;
  cout << "=========" << '\t' << "==========" << '\t' << "=========" << '\t' << "==========" << endl;

  for(south=southBoundary.begin(),north=northBoundary.begin();
      south!= southBoundary.end(),north!=northBoundary.end();++north,++south)
    cout << south->second << '\t' << '\t' << south->first << '\t' << '\t'
        << north->second << '\t' << '\t' << north->first << endl;

  return status;
}


int GridMap:: printEastWestBoundary() {

  int status = console->xstatus;
  console->MsgPrint("GridMap", "East West Boundary");
  if(westBoundary.size() == 0) {
    status = console->estatus;
    console->ErrMsg("GridMap","Empty West Boundary Map");
    console->ErrMsg("GridMap","Perhapse, you are calling this method before calling defineEastWestBoundaryCells method");
    return status;
  }
  
 
  map<int,int>::const_iterator west;
  map<int,int>::const_iterator east;
  
  cout << "Src West#" << '\t' << "Dest West#" << '\t' << "Src East#" << '\t' << "Dest East#" << endl;
  cout << "=========" << '\t' << "==========" << '\t' << "=========" << '\t' << "==========" << endl;

  for(west=westBoundary.begin(),east=eastBoundary.begin();
      west!= westBoundary.end(),east!=eastBoundary.end();++east,++west)
    cout << west->second << '\t' << '\t' << west->first << '\t' << '\t'
        << east->second << '\t' << '\t' << east->first << endl;
        
  return status;        
}


int GridMap::printClippedCells() {
  
  int status = console->xstatus;
  console->MsgPrint("GridMap", "Clipped Cells");

  // clipID: contains lower left(src=0,dest=1), upper right(2,3), lower right(4,5),
  // and upper left(6,7) cell numbers of the src an then
  // destination grid
  cout << endl;
  cout << "Source Grid Cells IDs Clipped to Destination Grid Cells" << endl;
  cout << "=======================================================" << endl;
  int sdist = abs(clipID[0] - clipID[4]);    // number of cols between left & right
  for(unsigned int i=0; i<srcClippedIDs.size();)  {
    for(int j=0; j<=sdist; j++)
      cout <<  srcClippedIDs[i++] << "  " ;
    cout << endl;
    cout << endl;
  }

  cout << endl;
  cout << "Destination Grid Cells Clipped to Source Grid Cells" << endl;
  cout << "===================================================" << endl;
  int ddist = abs(clipID[1] - clipID[5]);    // number of cols between left & right
  for(unsigned int i=0; i<destClippedIDs.size();)  {
    for(int j=0; j<=ddist; j++)
      cout <<  destClippedIDs[i++] << "  " ;
    cout << endl;
    cout << endl;
  }

  return status;
}


void GridMap::printGrids() {

   console->MsgPrint("GridMap","Source Grid");
   SourceGrid->print();
   console->MsgPrint("GridMap","Destination Grid");
   DestinationGrid->print();
}


void GridMap::drawGrid(string name) {

  int row, col, r, c;
  
  if(name == "SOURCE") {
    row = SourceGrid->getNumRow();
    col = SourceGrid->getNumCol();
  }

  else if(name == "DESTINATION") {
    row = DestinationGrid->getNumRow();
    col = DestinationGrid->getNumCol();
  }

  // for now
  int scell = 1;    //starting cell in grid unit of one, starting at 0
  int ecell = col;  // ending cell, cell ends at column
  //(int test = ecell - scell == col)
  static char star = '*', space = '+', endln = '\n';

  for (r = row; r >= 0; r--) {
    for (c = 0; c < scell; c++) 
      putc(space,stdout);
    for (c = scell; c<= ecell; c++)   // how many cells in each row
      putc(star,stdout);
    putc(endln,stdout);
  }
  cout << endl;
}


/* For each cell in the one2oneMap list of cell ids
 * draws each cell at its proper place in its grid
 * system. The larger grid is drawn beneath the smaller
 * grid.
 */
int GridMap::drawOne2OneMap() {

  int status = console->estatus;
  vector<int> src_ids, dest_ids;
  FILE* fptr;
  
  if((fptr = fopen("one2oneMap.out", "w")) == NULL) {
    cout << "Can not open one2oneMap.out file!" << endl ;
    return status;
  }
  
  if(DestinationGrid->getTimeStep() < SourceGrid->getTimeStep()) {
  
    // int he map itr->first is for dest which is smaller size
    map<int,int>::const_iterator itr;
    for(itr=one2oneMap.begin(); itr!=one2oneMap.end(); ++itr) {
      src_ids.push_back(itr->second);
      dest_ids.push_back(itr->first);
    } // end of for

    // make the src sort and get rid of repeated cells
    vector<int> srcid = filter(src_ids);
    
    // first draw the larger
    //SourceGrid->drawCell(srcid,fptr,star);
    //fclose(fptr);
    //if((fptr = fopen("one2oneMap.out", "a")) == NULL) {
    //  cout << "Can not append one2oneMap.out file!" << endl ;
    //  return status;
    //}
    
    DestinationGrid->drawCell(dest_ids,fptr);
    fclose(fptr);
  
  } // end of src larger
  return status;
}


/* This method writes the mapped grid cell information
 * into a file. The des. grid contains the row, col,
 * cell id, and the id of the source grid cell that
 * maps to the id of destination cell. 
 */
void GridMap::writeDestinationGrid(string outfile) {
  console->MsgWrite("GridMap", "Destination Grid",outfile);
  vector<string> header;
  header.push_back("This is finer grid ids mapped to coarser grid ids");
  header.push_back("Note that the mapped id = -1 means no cell intersect between 2 grids.");
  header.push_back("ID        ROW        COL       MAPPED_ID");
  header.push_back("=========================================");
  header.push_back("LINE BELOW IS RESERVED FOR APPLICATION USE, DEFUALT VALUE: ");
  DestinationGrid->writeGrid(outfile,header);
}


// decided to do this instead of making vector of
// map id, in case of bigger dest cell to smaller src cell!
int GridMap::writeSrcGrid(string outfile) {

  int status = console->xstatus;    // Error code variable
  
  console->MsgWrite("GridMap", "Source Grid",outfile);
  map<int,int>::const_iterator itr;
  multimap<int,int> many2oneMap;

  for(itr=one2manyMap.begin(); itr!=one2manyMap.end(); ++itr)
    many2oneMap.insert(make_pair(itr->second,itr->first));

  ofstream out;
 
  // Open the output file for writting
  out.open(outfile.c_str(), ios::out);
  if (!out) {
    status = console->estatus;
    console->MsgFail("GridMap","Open File");
    return status;
  }

  int day_step = 1825;   //default, reading so many days of data at one time.
  float tsd = DestinationGrid->getTimeStep();
  float tss = SourceGrid->getTimeStep();
  int ratio = (tss/tsd+1) * (tss/tsd+1);
 
  vector<Cell*> cells = SourceGrid->getCellList();
  multimap<int,int>::const_iterator mitr;
  int i=0;
  out << "This is coarser grid ids mapped to finer grid ids" << endl;
  out << "Note that the mapped id = -1 means no cell intersect between 2 grids." << endl;
  out << "ID" << '\t' << "ROW" << '\t' << "COL" << '\t' << "MAPPED_ID" << endl;
  out << "==" << '\t' << "===" << '\t' << "===" << '\t' << "=========" << endl;
  out << "LINE BELOW IS RESERVED FOR APPLICATION USE, DEFAULT VALUE: " << endl;

  // to be implemented, not uses yet
  vector<string> header;
  header.push_back("This is coarser grid ids mapped to finer grid ids");
  header.push_back("Note that the mapped id = -1 means no cell intersect between 2 grids.");
  header.push_back("ID        ROW        COL       MAPPED_ID");
  header.push_back("=========================================");
  header.push_back("LINE BELOW IS RESERVED FOR APPLICATION USE, DEFUALT VALUE: ");
  // end of to be implemented


  out << ratio << '\t'<< SourceGrid->getNumRow() << '\t'
      << SourceGrid->getNumCol() << '\t' << day_step << endl;
  for(mitr=many2oneMap.begin(); mitr!=many2oneMap.end();) {
    // fill the out of boundary src cells with -1
    if(mitr->first != cells[i]->getID()) {
      out << cells[i]->getID() << '\t'
          << cells[i]->getRow() << '\t'
          << cells[i]->getCol() << '\t'
          << cells[i]->getMappedID() << endl;
      i++; 
    }
    else {
      while (mitr->first == cells[i]->getID()) {
        out << mitr->first << '\t'
            << cells[i]->getRow() << '\t'
            << cells[i]->getCol() << '\t'
            << mitr->second << endl;
        mitr++;  //must be incr. here not in the for loop
      }
      i++;
    }
    if(mitr == many2oneMap.end()) {
      //finish the columns in the last row, is needed by different applications
      //it passes the dest. grid limit, however is marked by -1.

      int tmpcol = cells[i]->getCol();

      for(int j=tmpcol+1;j<=SourceGrid->getNumCol()+1;j++,i++)
        out << cells[i]->getID() << '\t'
            << cells[i]->getRow() << '\t'
            << cells[i]->getCol() << '\t'
            << cells[i]->getMappedID() << endl;
    }
  }

  out.close();
  return status;
}


/* This method creates a template for default
 * grids (elm and wmm) for the user. To use
 * this method run the program in command line:
 * prompt> grmap cfgfile.template
 * This is to recover the configuration file
 * for the format compatibility.
 * TO BE IMPLEMENTED!
 */
int GridMap::createDefaultCfgTemplate() {

  int status = console->xstatus;
  console->MsgWrite("GridMap","Default Configuration Template","grmap_template.cfg");
 
  FILE* fptr;

  if((fptr = fopen("grmap_template.cfg", "w")) == NULL) {
    console->MsgFail("GridMap","Open grmap_template.cfg");
    return console->estatus;
  }
        
  fprintf(fptr,"// This is a configuration file used for remapping and resampling binary\n");
  fprintf(fptr,"// data of any rectangulare grid (source grid with a squared shape cell)\n");
  fprintf(fptr,"// to another rectangular grid (destination grid with a squared shape cell).\n");
  fprintf(fptr,"// A square cell has equal time step in both direction (x,y).\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// The keywords used here must be typed exact and in the same order.\n");
  fprintf(fptr,"// All the keywords are typed in capital letters, including the pre-define\n");
  fprintf(fptr,"// values for x and y direction (UP, DOWN, LEFT, RIGHT). \n");
  fprintf(fptr,"// The operator (:=), prantesis and the terminators (;) are part of the format.\n");
  fprintf(fptr,"// The comments (//) are ignored and only '//' is a valid comment.\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// Any string after the ';' terminator is ignored. Therefore you don't\n");
  fprintf(fptr,"// need to comment the descriptions in the same line as the keywords.\n");
  fprintf(fptr,"// However, make sure if comments extend to the next line, comment\n");
  fprintf(fptr,"// that line.\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// No blank space between the lines are permitted. For readability,\n");
  fprintf(fptr,"// separate the lines with '//' comment.\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// The keyword 'NUMBERING' is used for numbering the cells, rows and columns\n");
  fprintf(fptr,"// start point. For example, if numbering is 1, the first row starts\n");
  fprintf(fptr,"// at 1 and the first column starts at 1 (1,1) and the cell number starts at 1. \n");
  fprintf(fptr,"// If numbering is 0 then the cell numbers start at 0, the rows and coulums \n");
  fprintf(fptr,"// start at (0,0). The cell numbers increas or decrease based on the direction.\n");
  fprintf(fptr,"// The numbering should not affect the number of rows and columns defined by \n");
  fprintf(fptr,"// the user. For example, ELM grid contains 184 rows and 108 columns if \n");
  fprintf(fptr,"// we count from 1. However, if we count from 0, ELM is 183x107 .\n");
  fprintf(fptr,"// Regardless of the numbering, the user must count the number of rows\n");
  fprintf(fptr,"// and columns from 1 and the program accounts for the differences.\n");
  fprintf(fptr,"// Destination and the src grid numbering can be different.\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// The number of rows and columns must be defined for a complete rectangular\n");
  fprintf(fptr,"// grid and not for a clipped grid such as sfwmm or nsm boundary grid.\n");
  fprintf(fptr,"// For example, NSM is a 80x41 grid system which has 3280 number of nodes.\n");
  fprintf(fptr,"// However, a grid_io binary file for NSM may have only 2328 nodes.\n");
  fprintf(fptr,"// The difference is the grid cells with no data. Therefore don't use \n");
  fprintf(fptr,"// the number of rows and number of nodes of a grid_io to calculate the\n");
  fprintf(fptr,"// number of columns. If one of the grids to be defined in this file is\n");
  fprintf(fptr,"// NSM grid, the NUMBERING=your choice (0,1,2,...), the ROWs=80, and the\n");
  fprintf(fptr,"// COLUMNs=41 .\n");
  fprintf(fptr,"//\n");
  fprintf(fptr,"// Data types are internally implemented as DOUBLE,FLOAT,INT, and STRING .\n");
  fprintf(fptr,"// For example the coordinates are assumed to be DOUBLE and direction \n");
  fprintf(fptr,"// is read as a STRING .\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"// Both grids must be in same datum and same unit, however, the origin\n");
  fprintf(fptr,"// and the directions can be different for each grid.\n");
  fprintf(fptr,"// \n");
  fprintf(fptr,"DESTINATION elm \n");
    fprintf(fptr,"//   KEYWORD  := (your value);               Description\n");
    fprintf(fptr,"//================================================================== \n");
    fprintf(fptr,"// Upper Left Coords in feet\n");
    fprintf(fptr,"TERM X_ORIGIN := (566672.42839 );         don't forget the terminator ;\n");
    fprintf(fptr,"TERM Y_ORIGIN := (861503.97229);          don't forget the () \n");
    fprintf(fptr,"TERM X_DIRECTION    := (RIGHT);           or LEFT, must be capital letter \n");
    fprintf(fptr,"TERM Y_DIRECTION    := (DOWN);            or DOWN, must be capital letter \n");
    fprintf(fptr,"TERM NUM_ROW  := (184);                   this is Elm grid number of rows \n");
    fprintf(fptr,"TERM NUM_COL  := (108);                   this is Elm grid number of cols \n");
    fprintf(fptr,"TERM TIMESTEP := (3280.833333);           for a square grid, same in x and y direction\n");
    fprintf(fptr,"TERM NUMBERING := (0);                    row col starts at 0,0 or 1,1 \n");
    fprintf(fptr,"// \n");
    fprintf(fptr,"// The follwing keyword is for drawing the grid and is OPTIONAL\n");
    fprintf(fptr,"// It has not been fully implemented yet. However, the drawing\n");
    fprintf(fptr,"// is available through few methods. See the src files\n");
    fprintf(fptr,"// TERM CELL_LOCATION := (1-5,1-3,1-6,5-3...); \n");
  fprintf(fptr,"END_DEST \n");
  fprintf(fptr,"SOURCE wmm \n");
    fprintf(fptr,"//   KEYWORD  := ( your value );\n");
    fprintf(fptr,"//================================================================== \n");
    fprintf(fptr,"// Lower Left Coords in feet \n");
    fprintf(fptr,"TERM X_ORIGIN := (548608.43463);         wmm model x origin \n");
    fprintf(fptr,"TERM Y_ORIGIN := (292037.72554); \n");
    fprintf(fptr,"TERM X_DIRECTION    := (RIGHT);\n");
    fprintf(fptr,"TERM Y_DIRECTION    := (UP);\n");
    fprintf(fptr,"TERM NUM_ROW  := (65);                   wmm rows \n");
    fprintf(fptr,"TERM NUM_COL  := (41);                   nsm/wmm cols \n");
    fprintf(fptr,"TERM TIMESTEP := (10560.0);              nsm/wmm/nexrad grid cell size \n");
    fprintf(fptr,"TERM NUMBERING := (1);                   nsm/wmm/nextrad \n");
  fprintf(fptr,"END_SRC \n");
  fclose(fptr);
  return status;
}

