/***************************************************************************
                          keywordmgr.cpp  -  description
                             -------------------
    begin                : Tue Feb 11 2003
    copyright            : (C) 2003 by Beheen Trimble
    email                : btrimble@sfwmd.gov
 ***************************************************************************/
/*  Note: All the methods in this class were originally written by Joseph
          Park (HSM Division) and they are taylored for this application.  */
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "keywordmgr.h"
#include "keywordinputblock.h"

/* This class manages another class called Keyword.
 * By reading a pre-defined format configuration file
 * the manager populates the data members of the Keyword
 * class and keeps a reference to those Keyword objects 
 * in its data member keyInputBlock. Other classes may
 * use the manager's data member through provided methods
 * of the manager.
 */
KeywordMgr::KeywordMgr(){
   console = new ConsoleMsg("KeywordMgr");
}

KeywordMgr::~KeywordMgr(){
  #ifdef DEBUG
  cout << "KeywordMgr ==> Deleting KeywordMgr " << endl;
  #endif
  
  console->~ConsoleMsg();
  map<string,KeywordInputBlock*>::const_iterator itr;
  for(itr=keyInputBlock.begin(); itr!=keyInputBlock.end(); ++itr) {
    KeywordInputBlock* k = itr->second;
    k->~KeywordInputBlock();
  }
}

/* This method is reading a configuration file and
 * saves the content in its data member fileVector.
 * The fileVector is used to populate the Keyword
 * objects.
 */  
int KeywordMgr::readCfgFile(std::string fileName) {

  ifstream fileStream;       // file stream access object

  int status = console->xstatus;    // Error code variable

  // Open the input file for reading
  fileStream.open(fileName.c_str(), ios::in);
  if (!fileStream) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Failed to open ", fileName, status);
    return status;
  }
  
  #ifdef DEBUG
            cout << "KeywordMgr ==> Reading configuration file " << fileName << endl;
            cout << "KeywordMgr ==> ===============================================" << endl;
  console->DebugMsg("KeywordMgr ==> Opened file ", fileName, status);
  #endif
  
  // Read each line into a string, store in vector
  // I would like to get an iterator that would terminate on \n instead
  // of whitespace as the istream_iterator does, then this vector creation
  // could be done all at once:
  // vector<string> fileVector (input_iterator, eos);
  // Since don't know how to do that, expand the vector as lines are read in.
  // 
  std::string thisLine = "";           // string to hold single line during read
  while(!fileStream.eof() and fileStream.good()) {
    getline(fileStream, thisLine);
    if(thisLine != "")
      fileVector.push_back(thisLine);
  }
  fileStream.close();
  
  #ifdef DEBUG
  console->DebugMsg("KeywordMgr ==> Closed file ", fileName, status);
            cout << "KeywordMgr ==> ================================" << endl;
  #endif
  
  // Dump read in file contents to console
  #ifdef DEBUG
            cout << "KeywordMgr ==> Printing configuration contents " << endl;
            cout << "KeywordMgr ==> ================================" << endl;
  vector<string>::iterator fileIterator = fileVector.begin();
  
  while(fileIterator != fileVector.end()) {
    console->DebugMsg("KeywordMgr ==> ", fileIterator->c_str(), status);
    fileIterator++;
  }
              cout << "KeywordMgr ==> ********************************" << endl;
  #endif
  
  return status;
}


/*
 * getting the start and end keywords from the keyword owner
 * this method populates one KeywordInputBlock object per
 * start and end keyword. The manager keep a reference to
 * these populated objects in its data member keyInputBlock.
 */
int KeywordMgr::parseInputBlock(map<string,Keyword*>& keywords,
                             string keyStart,string keyEnd) {

  int status = console->xstatus;

  string exStr;  // string for exception messages
  
  // Parse the variables between start and end keyword
  string fileLine;
  string::size_type stringKeyStart;
  string::size_type stringKeyEnd;
  vector<string>::iterator fileIterator = fileVector.begin();

  while(fileIterator != fileVector.end()) {
 
    fileLine = *fileIterator;
    // Ignore comment lines
    if(commentLine(&fileLine)) {
      fileIterator++;
      continue;
    }

    // In elm application config file format, never reaches this
    // end key.
    // Find the keyStarts and Keyends in the vector data
    stringKeyEnd = fileLine.find(keyEnd);
  
    if(stringKeyEnd != fileLine.npos ) {
      
      // found END_??
      fileIterator++; // don't continue looking for next keyStart block.
                      // in some cases there is no need to go to
                      // the end of vector! Yet some cases we do!
                      // Example: in fuzzy applications there are many
                      // of same keyStart but in elm (gridmap) app
                      // there are only unique keyStarts that they
                      // are being called separately.
      #ifdef DEBUG
      cout << "KeywordMgr ==> End of "<< fileLine << endl;
      cout << "KeywordMgr ==> *********************************************" << endl;
      #endif
      
      return status;
    }
    stringKeyStart = fileLine.find(keyStart);
    
    if(stringKeyStart != fileLine.npos ) {
      // found keyStart
      // Get the input variable name
      string variableName;
      string::size_type startVarNamePosition;
      string::size_type endVarNamePosition;

      // Find start of variable name
      startVarNamePosition = findNoWhiteSpace(fileLine,
                                              stringKeyStart + keyStart.length(),
                                              forward ); 
      // Find end of variable name
      endVarNamePosition = fileLine.length();
      endVarNamePosition = findNoWhiteSpace(fileLine, --endVarNamePosition, back);
      
      if(endVarNamePosition <= startVarNamePosition) {
        status = console->estatus;
        console->ErrMsg("KeywordMgr ==> Failed to find end of variable name",
               "in [" + fileLine + "]", status);
        return status;
      }
      // Assign the variable name string
      variableName = fileLine.substr(startVarNamePosition,
                                     endVarNamePosition - startVarNamePosition + 1);
      
      Keyword *tmpkey, *key;
      KeywordInputBlock* kib;
      string type;
      try {

        exStr = "Failed to create Keyword object for ";
        tmpkey = new Keyword();
        if(tmpkey == NULL) throw exStr ;
        tmpkey = keywords[keyStart];  // keyword
                  
        type = tmpkey->type;   //index and types are set before

        exStr = "Wrong type ";
        if(type == "STRING")
           tmpkey->svalue = variableName;
        else if(type == "DOUBLE")
          tmpkey->dvalue = atof(variableName.c_str());
        else if(type == "FLOAT")
          tmpkey->fvalue = atof(variableName.c_str());
        else if(type == "INT")
          tmpkey->ivalue = atoi(variableName.c_str());
        else throw exStr;

        #ifdef DEBUG
        console->DebugMsg("KeywordMgr ==> Found variable ",  variableName, status);
        #endif
        
        // populate the KeywordInputBlock
        exStr = "Failed to create KeywordInputBlock object for ";
        kib = new KeywordInputBlock();
        
        if(kib == NULL) throw exStr;                 // kib is
        exStr = "Failed to create Keyword object for ";
        key = new Keyword(*tmpkey);                  // the startkey and variableName
        if(key == NULL) throw exStr;
        kib->setKeyword(key);                        // and key has the value from above
        this->keyInputBlock[keyStart] = kib;         // update keyword manager's map
        #ifdef DEBUG
        console->DebugMsg("KeywordMgr ==> Created KeywordInputBlock object for ",  variableName, status);
        #endif
      }
      catch(string msg){
        status = console->estatus;
        console->ErrMsg("KeywordMgr ==> ", msg +
                 variableName + " in [" + fileLine + "]", status);
        return status;
      }   
      
      // iterate until keyEnd, create variable terms
      while(fileIterator != fileVector.end()) {
        
        string termName;
        string::size_type TermStartPosition;
        string::size_type startTermNamePosition;
        string::size_type endTermNamePosition;

        fileIterator++;
        fileLine = *fileIterator;
       
        // Ignore comment lines
        if(commentLine(&fileLine) ) {
          continue;
        }

        // Find start of TERM in string
        TermStartPosition = fileLine.find("TERM");  //check on case
        
        if(TermStartPosition == fileLine.npos ) {

          // make sure is not the last line, end key
          stringKeyEnd = fileLine.find(keyEnd);
          if(stringKeyEnd != fileLine.npos ) {
            // found END_??, already incremented, don't need object
            // for end key, however, if decrement, we terminate
            // on found end key, since for this application multiple
            // terms and keywords within a src or destination is not
            // allowed.
            fileIterator--;
            break;  
          }
          
          // at this point if TERM not found, there is a format
          // error in the config file. Can not retain the startKey
          // that is why we can not let extra lines between start
          // and keywords. Only comment is okay.
          status = console->estatus;
          console->ErrMsg("KeywordMgr ==> Format error ","in [" + fileLine + "]", status);
          return status;
        }
       
        // Find start of the term name, TERM is 4 chars long
        startTermNamePosition = findNoWhiteSpace(fileLine,
                                                 TermStartPosition + 4,
                                                 forward );
        // Find end of term name
        endTermNamePosition = fileLine.find(":=");
        if(endTermNamePosition <= startTermNamePosition ||
          (endTermNamePosition == fileLine.npos)) {
          status = console->estatus;
          console->ErrMsg("KeywordMgr ==> Failed to find delimeter := ",
                          "in [" + fileLine + "]", status);
          return status;
        } 
          
        endTermNamePosition = findNoWhiteSpace(fileLine, --endTermNamePosition, back);
        
        // don't test for fileLine.npos, never reches because of --endTerm...
        if(endTermNamePosition <= startTermNamePosition) {
          status = console->estatus;   
          console->ErrMsg("KeywordMgr ==> Failed to find end of term name",
                 "in [" + fileLine + "]", status);
          return status;
        }
        // Assign the variable name string
        termName = fileLine.substr(startTermNamePosition,
                                   endTermNamePosition - startTermNamePosition + 1);
        #ifdef DEBUG
        console->DebugMsg("KeywordMgr ==> Found TERM ", termName, status);
        #endif
        
        // Check for end of line delimeter ';'
        string::size_type endTermLinePosition = 0;
        endTermLinePosition = fileLine.find_first_of(";");
        
        if(endTermLinePosition == fileLine.npos ) {
          status = console->estatus;
          console->ErrMsg("KeywordMgr ==> Failed to find term end of line delimeter ';' for term",
                 termName + " in [" + fileLine + "]", status);
          return status;
        }

        //
        // Get the value of the term
        //
        string::size_type startTermValuePosition = 0;
        string::size_type endTermValuePosition   = 0;
        string::size_type currentEndTermPosition = 0;
 
        // Validate the term delimeters ()
        status = checkTermParen(&fileLine);
        
        if(status != console->xstatus) {
          console->ErrMsg("KeywordMgr ==> Failed to find valid term value delimeters '()' for term",
                 termName + " in [" + fileLine + "]", status);
          return status;
        }
        // Iterate on the value, each value is of the format := (value,value,...);
        while(currentEndTermPosition < endTermLinePosition ) {
          // Increment the position vars since the X.Y pairs don't start the line
          // that way it will work for iterating through the line to find subsequent X.Y pairs
          startTermValuePosition = fileLine.find_first_of("(", ++startTermValuePosition);
          endTermValuePosition   = fileLine.find_first_of(")", ++endTermValuePosition);
          
          if(startTermValuePosition == fileLine.npos or
             endTermValuePosition   == fileLine.npos ) {
            status = console->estatus;
            console->ErrMsg("KeywordMgr ==> Failed to find term value delimeters '()' for term",
                   termName + " in [" + fileLine + "]", status);
            return status;
          }

          
          // populate the keywords, in this case keyword = termName
          try {
   
            exStr = "Possible format error in config file! ";
            tmpkey = new Keyword();
            if((tmpkey = keywords[termName]) == NULL ) throw exStr;

            exStr = "Format error in configuration file for ";
            if(tmpkey->keyword != termName) throw exStr;

            exStr = "Type error in configuration file for ";
            string type = tmpkey->type; //types are set before
            
            //setprecision(6);
            string s = fileLine.substr(startTermValuePosition+1,endTermValuePosition-startTermValuePosition-1);
            //if s is more than one string, it gets tokenized in the
            // related class. Example the BIN_FILE keyword.
            if(type == "STRING")
              tmpkey->svalue = s;
            else if(type == "DOUBLE"){
              tmpkey->dvalue = atof(s.c_str());
            }
            else if(type == "FLOAT") {
              tmpkey->fvalue = atof(s.c_str());
            }
            else if(type == "INT") {
              tmpkey->ivalue = atoi(s.c_str());
            }                            
            else throw exStr;

            exStr = "Failed to create Keyword object for ";
            key = new Keyword(*tmpkey);  // a deep copy
            if(key == NULL) throw exStr;
            
            // Populate the KeywordInputTerm(s) for this KeywordInputBlock
            kib->setKeywordInputTerm(key);
            #ifdef DEBUG
            console->DebugMsg("KeywordMgr ==> Created KeywordInputTerm object for", termName, status);
            #endif
          }
          catch(string msg) { 
            status = console->estatus;
            console->ErrMsg("KeywordMgr ==> ", msg +
                   termName + " in [" + fileLine + "]", status);
            return status;
          }
          
          // Look for end of line
          currentEndTermPosition = findNoWhiteSpace(fileLine, endTermValuePosition + 1, forward);
          if(currentEndTermPosition == endTermLinePosition ) {
            break;
          }

        } // while ( currentEndTermPosition < endTermLinePosition ) 
        
      } // while (fileIterator != fileVector.end()) // iterate until END_FUZZIFY
      
    } // if(stringKeyStart != fileLine.npos ), found FUZZIFY
   
    fileIterator++;

  } // while(fileIterator != fileVector.end())

  #ifdef DEBUG
  console->DebugMsg("KeywordMgr ==> End of ","parseInputBlock", status);
  #endif
  
  return status;
} // end parse
  

/* Given a map and a string, this method finds and returns
 * the value of the given key or returns null, if not found.
 */
Keyword* KeywordMgr::findKeywordFromMap(string key,
                                        map<string,Keyword*>& thismap) {
  int status = console->xstatus;
  
  if(thismap.find(key) == thismap.end() ) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Failed to find keyword in map" , key, status);
    return NULL;
  }

  return thismap[key];
}


// findNoWhiteSpace
string::size_type KeywordMgr::findNoWhiteSpace(string str,
                                               string::size_type offset,
                                               int direction ) {
  int status = console->xstatus;
  
  string::size_type pos = offset;
  string::size_type eos = str.length();

  if(direction == forward) {
    while(str[pos] == ' ' || str[pos] == '\t' ||
          str[pos] == '\r' || str[pos] == '\n') {
      if (pos >= eos ) return eos;
      pos++;
    }
  }
  else if(direction == back) {
    while(str[pos] == ' ' || str[pos] == '\t' ||
          str[pos] == '\r' || str[pos] == '\n') {
      if (pos <= 0 ) return 0;
      pos--;
    }
  }
  else {
    pos = 0;
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> findNoWhiteSpace invalid direction " , str, status);
    // don't return status
  }

  return pos;
}



// Assumes that "//" is the START of the comment line
// A line that has any non-whitespace before "//" will
// not be recognized as a comment line.
// Note that the FCL comment is apparently (* *)
//
bool KeywordMgr::commentLine(string *line) {

  string::size_type commentPosition = 0;
  // Find first non-whitespace in line
  commentPosition = findNoWhiteSpace(*line, 0, forward );
  if(line->substr(commentPosition, 2) == "//" ) {
    return true;
  }
  if(line->substr(commentPosition, 2) == "(*" ) {
    return true;
  }
  return false;
}


// getLine
//
// Purpose: Form a complete line by looking for ';'
//
// Arguments: fileIterator, fileLine
//
// Return:   string line filled with text to ';'
//           int number of iterations to apply to fileIterator
//           (the number of physical lines that were read in.)
//
int KeywordMgr::getLine(vector<string>::iterator fileIterator,
                        string* line ) {
  int increment = 0;

  string newLine = *fileIterator;
  string::size_type EOLPosition = fileIterator->find(';');

  while(commentLine(&(*fileIterator))) {
    fileIterator++;
    increment++;
  }

  newLine = *fileIterator;
  EOLPosition = fileIterator->find(';');

  while(EOLPosition == fileIterator->npos ) {
    fileIterator++;
    increment++;
    if(commentLine(&(*fileIterator))) {
      newLine = "";
      continue;
    }
    if(fileIterator == fileVector.end()) {
      increment = -1;   // not estatus
      return increment;
    }
    EOLPosition = fileIterator->find(';');
    newLine = newLine + " " + *fileIterator;
  }

  *line = newLine;

#ifdef DEBUG
  cout << "KeywordMgr ==> getLine("<< increment << ") : [" << *line << "]\n";
#endif

  return increment;
}


int KeywordMgr::checkTermParen(string* inString) {

  int status = console->xstatus;
  
  string::size_type pos = 0;
  string::size_type eos = 0;
  char firstParen  = 0;
  char secondParen = 0;

  bool foundFirst  = false;
  bool foundSecond = false;

  string localString = *inString;

  eos = localString.length();

  // Validate that (..) occur in sequence
  while(pos <= eos ) {
    if(localString[pos] == '(' or localString[pos] == ')' ) {
      if(not foundFirst ) {
        foundFirst = true;
        firstParen = localString[pos];
      }
      else if(foundFirst and not foundSecond ) {
        foundSecond = true;
        secondParen = localString[pos];
        if(firstParen == secondParen ) {
          status =  console->xstatus;
          return status;
        }
      }
      if(foundFirst and foundSecond ) {
        foundFirst  = false;
        foundSecond = false;
      }
    }
    pos++;
  }
  // validate that there are even numbers of ( )
  if(foundFirst and not foundSecond ) {
    status = console->estatus;
    return status;
  }

  return status;
}

// Checks only on the keyStart and keyEnd.
// Translate them to uppercase (must be implemented)
// Checks for delimeters, whitespace, parantesis and term names
// are done during parsing.
//
int KeywordMgr::checkFormat(string keyStart,string keyEnd) {

  int status = console->xstatus;
  int istart  = 0;
  int iend = 0;


  string fileLine;
  string::size_type startPosition;
  string::size_type endPosition;

  vector<string>::iterator fileIterator = fileVector.begin();

  while(fileIterator != fileVector.end()) {
    fileLine = *fileIterator;
    // to be implemented for ignoring upper or lower
    // case entries by the user.
    // string  s = fileLine.upcase();  // not available
    // fileVector[len].c_str; //.upcase();
    // cout << "s is " << s << endl;

    // Ignore comment lines
    if(commentLine(&fileLine)) {
      fileIterator++;
      continue;
    }

    // Be cautious on using the 'find' method.
    // For example if a key = "XY" and there
    // is another key that is called "END_XY"
    // the find method sees them as the same.
    
    startPosition = fileLine.find(keyStart);
  
    endPosition = fileLine.find(keyEnd);
   
    if(startPosition != fileLine.npos) {
      // found keyStart
      // int len = startPosition + keyStart.length();
      istart++;
    }
    if(endPosition != fileLine.npos ) {
      // found keyEnd
      iend++;
    }
    fileIterator++;
  } // finished the file vector
  
  if(not istart) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Failed to find ",keyStart + " in config file", status);
    return status;
  }
  if(not iend) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Failed to find ",keyEnd + " in config file", status);
    return status;
  }
  if(istart > 1 ) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Too many ",keyStart + " blocks in the file", status);
    return status;
  }
  if(iend > 1 ) {
    status = console->estatus;
    console->ErrMsg("KeywordMgr ==> Too many ",keyEnd + " blocks in the file", status);
    return status;
  }

  return status;

} // end format

/* This method deligates the print to its data member
 * keyInputBlock. In returns all the keywords are
 * printed to the screen.
 */
void KeywordMgr::printInputBlock() {

  map<string,KeywordInputBlock*>::const_iterator itr;
  int status = console->xstatus;
  
  for(itr = keyInputBlock.begin(); itr != keyInputBlock.end(); ++itr) {
    
    // Note that map->first returns the key, map->second returns the value
    string mapkey = itr->first;
    console->Msg("KeywordMgr ==> ", "Printing keyword input block for " + mapkey, status);
         cout << "KeywordMgr ==> ===================================" << endl;
    KeywordInputBlock* blk = itr->second;
    blk->printKeyword();
    blk->printKeywordTerm();
  }
}

/* The manager has a reference to one or more block objects,
 * given the name of the block, key, it finds the requested
 * block and its term objects and returns them in their
 * internal form, a keyword object.
 * Returns null otherwise.
 */
map<string,Keyword*> KeywordMgr::getInputBlock(string key) {

    Keyword* ky;
    map<string,Keyword*> returnMap;
    vector<Keyword*> returnTerm;
    
    KeywordInputBlock* inblock = keyInputBlock[key];
    if(inblock) {                  
      ky = inblock->getKeyword();
      returnMap[ky->keyword]=ky;

      returnTerm = inblock->getInputTermKeys();
      
      for(unsigned int j=0; j<returnTerm.size(); ++j) {
        ky = returnTerm[j];
        returnMap[ky->keyword] = ky;
      }
      return returnMap;
    }

    else {
      returnMap[key]=NULL;
      return returnMap;
    }
}

/*
 * This method finds an input block by its name and updates
 * its input terms from the update list.
 */
int KeywordMgr::updateInputBlock(string key, vector<Keyword*> updatelist) {

  int status = console->xstatus;
  
  KeywordInputBlock* inblock = keyInputBlock[key];

  if(inblock) {

    Keyword* ky = inblock->getKeyword();
    ky->printKeyword();
   
    for(unsigned int i=0; i<updatelist.size(); ++i) {
      Keyword* thiskey = updatelist[i];
      
      // Populate the KeywordInputTerm(s) for this KeywordInputBlock
      inblock->setKeywordInputTerm(thiskey);
      string term = thiskey->keyword;
      #ifdef DEBUG
      console->DebugMsg("KeywordMgr ==> Created KeywordInputTerm object for", term, status);
      #endif      
    }
    return status;
  }
  else {
    status = console->estatus;
    return status;
  }
}
