/*! \file driver_utilities.h
\brief Header file for Driver Utilities. 

This defines or declares variables & functions that are global to Driver_Utilities.c.  \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Dec 2004 \n
*/

/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Re-organized, clarified scopes, Doxygen tags added 
	
*/

#ifndef DRIVER_UTILITIES_H
#define DRIVER_UTILITIES_H

#include "globals.h"


#define Outlist_size 250	/*!< The (maximum) number of variables in model outlist */

              /* TODO: develop flexible output of calendar-based and julian-based outsteps (jan 11, 2005) */
              /* kluge here in defining the int for the Model.outList that will denote calendar-month output */
int CalMonOut=99999;  /*!< The outstep interval (in Model.outList input file) used to denote output intervals of 1-calendar-month instead of selectable-julian-day */

/*! \anchor environmentVariables
	\par Environment variables used in model
	\n\em ModelPath environment variable, base pathname for executable and input data 
	\n\em ProjName environment variable, the name of the model project
	\n\em DriverPath environment variable, base pathname for source code
	\n\em OS_TYPE environment variable, the type of operating system being used (informational purpose only, not used in code) */
char *ModelPath, *ProjName, *DriverPath, *OS_TYPE; 

char *OutputPath;	/*!< base pathname for all model output (user input) */

/*! \anchor ModelNameVersion
	\par Model name/version (user input)
	\n\em modelName Name given to model implementation (user input)
	\n\em modelVers Version given to model implementation (e.g., v.2.1) (user input) */
char modelName[20], modelVers[10];    

char modelFileName[300]; 	/*!< Generic string for input/output file name (overwritten/reused) */

int gNPtTs;
int PListIndex=0;
int total_memory=0;
float *plot_array, gTS;
	
int gridSize;	/*!< row x col size of model grid domain, adding 2 to each dimension for local grid buffer (intended primarily for parallel implementation) */
int gTempSize;	/*!< intermediate variable for model grid memory alloc */
UCHAR* gTemp;	/*!< intermediate variable for model grid memory alloc */


VOIDP nalloc(unsigned mem_size, const char var_name[]);
void init_pvar(VOIDP Map, UCHAR* mask, unsigned char Mtype,float iv);
void read_map_dims(const char* filename);

void read_model_parameters(char* s_parm_name, int s_parm_relval);

float* get_DBase_parmOLD(const char *filename, int i0, int i1, int index);
float* get_DBase_parm(const char *filename, float *dArray);

ViewParm *read_output_parms(void);     
ViewParm *readOutlist(int size); 

int readViewParms(FILE *vpFileName, int size, ViewParm **ppview); 
void setPrecision(ViewParm* vp, int value);
int write_map_file(const char* filename, VOIDP Map, char Mtype,
                   int index, float scale_value, float offset_value,
                   int bSize, unsigned char type);
int read_map_file(const char* filename, VOIDP Map,
		  unsigned char Mtype, float scale_value,
		  float offset_value);
void print_loc_ave(Point3D *vt, void* Map, char Mtype, char* mName, int tIndex);
void write_output(int index, ViewParm* vp, void* Map, const char* filename, char Mtype, int step);
void print_point(ViewParm *vp, void* Map, char Mtype, char* mName, int tIndex, int vpindex);
void writeWindow(void* fValue, char* vName, char* desc, int x0, int y0, int N0, int N1, int index, UCHAR Mtype, UCHAR format);
void calc_maxmin(ViewParm *vp, void* Map, char Mtype, char* mName, int step);
void quick_look(void* Map, char* name, unsigned char Mtype, int xc, int yc, int range, unsigned char format);
int iclip ( int x0, int min, int max );
void make_more_points(ViewParm *vp, int ptIncr);
void enc_Nb(UCHAR** sptr,SLONG value,int bytes);


void RP_SwapFields (RPoint* thisStruct, RPoint* that);
void RP_CopyFields(RPoint* thisStruct, RPoint* that);
void RP_SetFields(RPoint* thisStruct, int ix, int iy, float r, float value);
void RPL_AddrPoint(RPointList *thisStruct, int x, int y, float r, float value);
void RPL_Sort(RPointList *thisStruct);
void RPL_Free(RPointList* thisStruct);
void RPL_AddrPoint(RPointList *thisStruct, int x, int y, float r, float value);
RPointList* RPL_Init(int nPoints);

void PTS_CopyFields(PTSeries* thisStruct, PTSeries pV);
void PTS_Free(PTSeries* thisStruct);
void PTS_SetFields (PTSeries* thisStruct, int ix, int iy,
                    int index, int format, int col);
void PTSL_AddpTSeries(PTSeriesList* thisStruct, int x, int y, int index, int seriesNum, int col);
void PTSL_Free(PTSeriesList* thisStruct);
void PTSL_ReadLists(PTSeriesList* thisStruct, const char *ptsFileName,
		    int index, float* timeStep, int* nPtTS, int col);
void PTSL_CreatePointMap(PTSeriesList* pList,void* Map, unsigned char Mtype,
	                 int step, float scale);
float PTSL_GetInterpolatedValue1(PTSeriesList* thisStruct, int x, int y, int step);
float PTSL_GetInterpolatedValue0(PTSeriesList* thisStruct, int x, int y, int step);
PTSeriesList* PTSL_Init(int nSlots, int format);

void calcdate(double jd, int *m, int *d, int *y, int *h, int *mi, double *sec);
double julday(int mon, int day, int year, int h, int mi, double se);
float FMOD( float x, float y);

void setup_grid();
void set_env_vars(void);
void set_boundary(VOIDP Map, unsigned char Mtype,float bv);

void fatal(const char *msg);
void Exit(int code);
void Copy(void *src, void *dst, int w, int n, int sw, int dw);
void getInt(FILE* inFile, const char* lString, int* iValPtr);
void getChar(FILE* inFile, const char* lString, char* cValPtr);
void getString(FILE* inFile, const char* lString, char* inString);
void getFloat(FILE* inFile, const char* lString, float* fValPtr );
void setFlag(ViewParm* vp, UINT mask, int value);
int getFlag(ViewParm* vp, UINT mask);
int getPrecision(ViewParm* vp);
int isInteger(char *target_str);
int isFloat(char *target_str);
int skip_white(FILE* infile);
int scan_forward (FILE* infile, const char* tstring);
int check_for (char *s0, const char *s1, int start, int  cs, int rp);
int find_char(FILE *infile, char tchar);
int Round(float x);
char* name_from_path(char* name);
char *Scip(char *s, char SYM );

/* below are unused or effectively unused  */
void setup_platform();
void link_edges(VOIDP Map, unsigned char Mtype);

float Normal(float mean,float sd);
int Poisson(float mu); 
/* above are unused or effectively unused  */

/* externs */
/* from serial.h */
extern char gCArg[kCArgDepth][kCArgWidth];
extern FILE* Driver_outfile;

extern int init_config_file(FILE *vpFile, char term1, char term2, char term3, char term4);
extern int parse_packet( FILE *vpFile, int* nArgs, char* test);
extern void Combine(float* fValue, char* label, int nComp, int cType, int step);
extern void writeMap(char*, void*, int, unsigned char, int);
extern void writeSeries(void* fValue, char* label, char* desc, int N0, int N1, byte Mtype, byte format);
extern float* readSeriesCol(char *filename, int format, int index, int* Npt, float* TStep, int col);
extern byte readMap(char*, void*);

/* more from serial.h: parallel-related stuff, effectively not functioning & ~undocumented for parallel */
extern int on_this_proc(int x,int y);
extern void exchange_borders(UCHAR* map, int size);
extern void exparam( struct nodenv* envInfo);
extern void exgridsplit(int nprocs, int ndim,int nprocs2[2]);
extern void exgridcoord(int pnum, int rnum[2]);
extern void exgridsize( int pnum, int gsize[2],int lsize[2],int lstart[2]);
extern void broadcastInt(int* iValPtr);
extern int exgridinit(int dim, int* nprocs);
extern void broadcastMsg( UCHAR* msgPtr);
extern void fasync(FILE* file);
extern void fmulti(FILE* file);

/* from generic_driver.h */
extern float avg_Intvl; /* used here as part of a TODO and kluge */
extern SeriesParm pSeries[MAX_PTSERIES];
extern ViewParm *view ;
extern Point2D dbgPt;
extern struct nodenv env;

extern int gbl_size[2], lcl_size[2], lcl_start[2];
extern int N_iter, istep;
extern int procnum, Lprocnum, nprocs[2], recpnum[2];
extern float gRTable[];

/* from unitmod.h */
//extern int ReadParms(FILE *parmFile);
extern int ReadGlobalParms(char* s_parm_name, int s_parm_relval);
extern int ReadHabParms(char* s_parm_name, int s_parm_relval);
extern float GP_IDW_pow;

/* from budgstats.h */
extern int numBasn;


#endif  /* DRIVER_UTILITIES_H */
