/*! \file generic_driver.h
\brief Header file for Generic Driver

This defines or declares variables & functions that are global to Generic_Driver.c. \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Re-organized, clarified scopes, Doxygen tags added 
       Jan 2005 v2.4.0: multi-run capability
       Apr 2005 v2.4.4: added (actually redid) flexible canal output
	
*/

#ifndef GENERIC_DRIVER_H
#define GENERIC_DRIVER_H

#include "globals.h"

int NSector;	/*!< number of Sectors (dynamic cell_dyn modules) */
int iSector[MAX_SECTOR];	/*!< sector (dynamic cell_dyn module) number */
int seed;	/*!<the seed value for (pseudo-) random number generator (unused) */
int ESPmodeON=0;     /*!< boolean flag indicating Everglades Settling-of Phosphorus mode. A mode with all biol/chem (non-hydro) modules
	    turned off, running only a net settling rate module that reproduces equations and data from the SFWMD's
	    old Everglades Water Quality Model (EWQM).  Done only for comparision to full ecological ELM - ESP doesn't work very well!!  */                       
int WatMgmtOn;	/*!< boolean flag indicating water management mode */
int HabSwitchOn;	/*!< boolean flag indicating habitat succession mode */
int PositAnalOn=0;	/*!< flag for Position Analysis mode:  for probability analysis of near-term water management options, 
					Position Analysis mode involves re-inititialization of model on particular month/day at recurring yearly interval */                          
char SimAlt[20]; 	/*!< simulation scenario/alterative name */
char SimModif[20];     /*!< simulation scenario/alterative modifier name/note */   
char outpath[120];     /*!< base pathname for all output */              

int gbl_size[2];	/*!< row [0] and column [1] dimensions of model grid domain, serial implementation */

char initDateRead[15];	/*!< calendar date of simulation initialization */
double Jdate_init;  /*!< julian day (since epochs ago) of the initialization of model */
double Jdate_end;  /*!< julian day (since epochs ago) of the ending of model */
	/*! \anchor DateGregor
		\par Variables associated with the gregorian calendar
 	   \n\em yr_in calendar year of initialization of model
 	   \n\em mo_in calendar month of initialization of model
 	   \n\em da_in calendar day of initialization of model
 	   \n\em yr_end calendar year of ending of model
 	   \n\em mo_end calendar month of ending of model
 	   \n\em da_end calendar day of ending of model 
 	   \n\em hr_in hour of initialization of model (unused)
 	   \n\em mi_in min of initialization of model (unused)
 	   \n\em se_in seconds of initialization of model (unused) */
int yr_in, mo_in, da_in, hr_in, mi_in, se_in, yr_end, mo_end, da_end;    
int mo_R_in;    /*!< month of Re-initialization in Position analysis mode */
int da_R_in;	/*!< day of Re-initialization in Position analysis mode */
int PORnumday;     /*!< number of days of simulation Period Of Record (incl. start day)  */                      
		
int N_iter;	/*!< count of maximum number of model time iterations */
int istep;	/*!< counter for number of model time iterations */
float step_Cell; 	/*!< constant used in horizontal surface water raster flux equations ( m^(-1.5) * sec )*/

float avg_Intvl=0;     /*!< time (day) interval between recurring-average summaries */                  
float budg_Intvl=0;    /*!< time (day) interval between budget summaries */                   
float BIRavg_Intvl=0;    /*!< time (day) interval between Basin/Indicator-Region stat summaries */                   
float can_Intvl=0;    /*!< time (day) interval between canal (water management) summaries */                   

int budgCalendar; 	/*!< boolean flag signifying use of gregorian calendar for budget calcs */
int avgCalendar = 0; 	/*!< boolean flag signifying use of gregorian calendar for recurring-average calcs */
int canalCalendar = 0; 	/*!< boolean flag signifying use of gregorian calendar for canal data output */
int avgPrint = 0; 	/*!< boolean flag to indicate if recurring-averages is to be printed at current time */
int canPrint = 0; 	/*!< boolean flag to indicate if canal data is to be printed at current time */

#define numRunSets 150
/* struct and pointer to struct for sensitivity analysis */
ProgAttr *ProgExec;
ProgAttr **RunList;

float gRTable[] =
        {.5000,.5398,.5793,.6179,.6554,.6915,.7257,.7580,.7881,.8159,
         .8413,.8643,.8849,.9032,.9192,.9332,.9452,.9554,.9641,.9713,
         .9773,.9821,.9861,.9893,.9918,.9938,.9953,.9965,.9974,.9981,
         .9987,.9990,.9993,.9995,.9997,.9998,.9998,.9999,.9999,1.00} ;	/*!< data used in function Normal, which is unused */

ViewParm *view;
SeriesParm pSeries[MAX_PTSERIES];	/*!< An array of SeriesParm structs for point time series output */
Point2D dbgPt;	/*!< struct of type Point2D, the grid location of a point */

struct nodenv env;	/*!< struct of nodenv, with parallel code info */

	/*! \anchor parallelData
		\par Data for parallel implementation, basically ignore
 	   \n\em Lprocnum NA for serial implementation (=1)
 	   \n\em procnum NA for serial implementation  (also see nodenv struct)
 	   \n\em nprocs NA for serial implementation (also see nodenv struct)
 	   \n\em recpnum NA for serial implementation
 	   \n\em tramType NA for serial implementation
 	   \n\em tramNum NA for serial implementation
 	   \n\em lcl_size used in serial code, but not really pertinent: in serial implementation, rows lcl_size[0]=s0 and columns lcl_size[1]=s1, dimensions of model grid domain local to one processor (for parallel implementation) 
 	   \n\em lcl_start used in serial code, but not really pertinent: in serial implementation, starting row lcl_start[0]=0 and starting column lcl_start[1]=0  in model grid domain local to one processor (for parallel implementation) */
int procnum, Lprocnum, nprocs[2], recpnum[2], tramType, tramNum[2], lcl_size[2], lcl_start[2];


void get_parmf();
void setup();
void track_time(int istep);

/* from multi_run.h */
//extern void get_SensiParms(char* s_parm_name);

/* from driver_utilities.h */
extern char *ModelPath, *ProjName, *OS_TYPE;
extern char *OutputPath;
extern char modelName[20], modelVers[10];

extern ViewParm *read_output_parms(void);

extern void read_model_parameters(char* s_parm_name, int s_parm_relval);

extern void getInt(FILE* inFile, const char* lString, int* iValPtr);
extern void getFloat(FILE* inFile, const char* lString, float* fValPtr );
extern void getString(FILE* inFile, const char* lString, char* inString);
extern void calcdate(double jd, int *m, int *d, int *y, int *h, int *mi, double *sec);
extern void setup_platform();
extern void set_env_vars(void);
extern void setup_grid();
extern int skip_white(FILE* infile);
extern int scan_forward (FILE* infile, const char* tstring);
extern double julday(int mon, int day, int year, int h, int mi, double se);
extern float FMOD( float x, float y); 

/* from serial.h */
extern void local_setup(int argc, char** argv);
extern void open_debug_outFile(int index);
extern void broadcastInt(int* iValPtr);
extern void sync_processors();
extern void send_point_lists2(SeriesParm *pSeries, int nSeries);
extern void open_point_lists(SeriesParm *pSeries, int nSeries); 

/* from unitmod.h */
extern void init_static_data(void);
extern void init_dynam_data(void);
extern void init_eqns(void);
extern void init_canals(int runNumb);
extern void init_succession(void);
extern void reinitBIR(void);
extern void reinitCanals(void);
extern void alloc_memory();
extern void gen_output(int step, ViewParm *view);
extern void get_map_dims(void);
extern int call_cell_dyn(int sector,int step); 

#endif  /* GENERIC_DRIVER_H */
