/*! \file watmgmt.h
\brief Header file for the Water Managment calculations. 

This defines or declares variables & functions that are global to WatMgmt.c.  \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

/* General notes on revisions to this source file. 
       Jan 2005 v2.3.3: documentation upgrade 
       		- Re-organized, clarified scopes, Doxygen tags added 
       May 2006 v2.5.1: added edgeMann parameter to struct Chan
	
*/
#ifndef WATMGM_H
#define WATMGM_H

#include "globals.h"

#define STEP(x) ( ((x)>=0) ? 1.0 : 0.0 )
#define MAX_H_STRUCT 180	/*!< maximum number of water control structures that have historical or other data as input */

#define LEFT 1
#define RIGHT -1

#define NONE 0
#define EAST 1
#define SOUTH 2
#define ALL 3

FILE *schedFile; 			/*!< File pointer to the schedule data file */
FILE *ChanInFile;  			/*!< File pointer to several (opened-closed) input files */
FILE *CanalOutFile;  		/*!< File pointer to canal hydrologic output file */
FILE *CanalOutFile_P;  		/*!< File pointer to canal phosphorus output file */
FILE *CanalOutFile_S;  		/*!< File pointer to canal salt/tracer output file */
FILE *WstructOutFile;  		/*!< File pointer to water control structure hydrologic output file */
FILE *WstructOutFile_P;  	/*!< File pointer to water control structure phosphorus output file */
FILE *WstructOutFile_S;  	/*!< File pointer to water control structure salt/tracer output file */
FILE *canDebugFile;   		/*!< File pointer to canal debug file */
FILE *CanalCellInteractDebug; /*!< File pointer to canal-cell interaction file */
FILE *F_struct_wat;			/*!< File pointer to the water control structure flow data file */
FILE *F_struct_TP;			/*!< File pointer to the water control structure TP data file */
FILE *F_struct_TS;			/*!< File pointer to the water control structure TS (salt/tracer) data file */

int CHo;						/*!< the number of the first canal in Chan.file sequence*/
int CHANNEL_MAX_ITER;	        /*!< maximal number of iterations in relaxation procedure (input data) */
int UTM_EOrigin, UTM_NOrigin;	/*!< Map origin, easting and northing UTM */
int printchan = 0; 				/*!< flag to indicate time to print canal/struct data */
int N_c_iter = 0;     	        /*!< cumulative number of iterations through canal code */
int num_cell = 0;     	        /*!< counter of number cells along a reach segment */
int C_Mark = 0;     	        /*!< indicator that a cell is marked with a canal already */

float C_F; 				/*!< acceleration of overland flow for canal/cell interactions - for sensitivity experiments only */
float F_ERROR; 			/*!< convergence error for relaxation procedure in canals */
float MINDEPTH;  		/*!< minimal allowed water depth in canal */

int num_chan; 			/*!< number of channels (canals) */
int num_struct_hist;	/*!< number of structures driven by data */ 
int numTPser;			/*!< number of structures with hist TP data */ 
int numTSser;			/*!< number of structures hist TS (TotalSalt/tracer) data */ 

float *MCopen;           /*!< manning's coef for open water, fixed parameter for all domain (used in structure flow for bridges) */
char modelFileName[300]; /*!< generic string for input/output file name */

/*! Static data definitions of a canal reach loc */
struct Chan_reach
{
 float x0,y0,x1,y1;                  /*!< UTM coordinates of beginning and end of canal reach */
 struct Chan_reach *next_reach;      /*!< pointer to next canal reach */
};  

/*! Static and dynamic data definitions of a canal */
struct Chan 		    
{
  int number;           /*!< numeric-ID of canal */
  int levee;         	/*!< levee location relative to canal: 1 =levee on the left, -1 =levee on the right, 2 =levees on both sides, 0 =no levees */
  int roil;		    	/*!< number of cells to left away from canal that are included in those interacting with canal - ALWAYS=1 in ELM */
  int roir;	            /*!< number of cells to right away from canal that are included in those interacting with canal - ALWAYS=1 in ELM */
  float depth;		    /*!< canal (not water) depth (m) */
  float width;		    /*!< canal width (m) */
  float cond;		    /*!< Levee hydraulic conductivity (seepage coefficient) for the canal (m/d) */
  double N;		    	/*!< UNUSED nitrogen mass in canal (kg) */
  double P;		    	/*!< phosphorus mass in canal  (kg) */
  double S;		    	/*!< salt mass in canal  (kg) */
  double N_con;		    /*!< UNUSED nitrogen concentration in canal  (mg/L = g/m3) */
  double P_con;		    /*!< phosphorus concentration in canal  (mg/L = g/m3) */
  double S_con;		    /*!< salt concentration in canal  (mg/L = g/m3) */
  float wat_depth;	    /*!< water depth in canal  (m) */

  float ic_N_con;	    /*!< UNUSED initial nitrogen concentration in canal  (mg/L = g/m3) */
  float ic_P_con;	    /*!< initial phosphorus concentration in canal  (mg/L = g/m3) */
  float ic_S_con;	    /*!< initial tracer concentration in canal  (g/L = kg/m3) */
  float ic_depth;	    /*!< initial water depth in canal  (m) */
  float edgeMann;	    /*!< Manning's n associated w/ edge of canal, to accomodate topographic lip/berm and/or denser veg along canal length; value of 0.00 has no effect  (d/(m^(1/3) ) */

  float length;		    /*!< length of canal (m) */
  float area;		    /*!< area of canal (m^2) */
  float elev_start;	    /*!< land elevation at starting point of canal (m NGVD 1929) */
  float elev_end;	    /*!< land elevation at ending point of canal (m NGVD 1929) */
  float elev_drop;	    /*!< land elevation drop between start&end points of canal reach (m) */
  float elev_slope;	    /*!< slope of the elevation gradient from start to end points  (m/m) */
  float minVol;		    /*!< minimum volume allowed in canal  (m^3) */
  float seg_len;	    /*!< avg length of cell along  segment (m) */
  float SW_flow_coef;	    /*!< (aggregated) overland flow coef (not incl. manning's n) (m^0.5 * sec/d) */
  float SPG_flow_coef;	    /*!< seepage coef modified by calib parm (m/d) */
  int   num_of_cells;       /*!< number of interacting cell patterns (pairs in this case) */
  int   basin;    	    /*!< hydro basin ID to which the canal  belongs */
  int parent;               /*!< ID if this is a main (parent) hydro basin */
  int family;               /*!< the family name (parent basin & IRegion children) of hydrologic basin */
  double sumHistOut;	    /*!< sum of all historical/other-data flows (corrected for excess demand) from a canal during an iteration (m^3/d) */
  double sumHistIn;	    /*!< sum of all historical/other-data flows (corrected for excess demand) to a canal during an iteration (m^3/d) */
  double sumRuleOut;	    /*!< sum of all rule-based flows from a canal during an iteration (m^3/d) */
  double sumRuleIn;	    /*!< sum of all rule-based flows to a canal during an iteration (m^3/d) */
  struct Chan_reach *reaches;    /*!< pointer to list of canal reaches within a canal */
  struct Cells *cells;           /*!< pointer to the list of adjacent cells */
  struct Chan *next_in_list;     /*!< points to the next canal reach in list */
};
struct Chan **Chan_list;
  

/*! Defines the cells adjacent to a canal */
struct Cells			 
{
 int x;				 /*!< row of the cell */
 int y;				 /*!< column of the cell */
 int ind;			 /*!< cell index = 1 for left cells and -1 for right cells */
 float length;			 /*!< length (m) of canal reach associated with this cell */
 float reachElev;			 /*!< elevation (m) of canal reach at location associated with this cell */
 float reachDistDown;			 /*!< from the beginning point of reach, the cumulative distance (m) of canal reach associated with this cell */
 struct Cells *next_cell;	 /*!< next pair of cells adjacent to the canal */  
};
struct Cells *cell_last;
struct Cells *cell;
 

/*! \brief Water control structure attributes */
struct Structure		
{ 
 int flag;                      /*!< historical/other-data data flag: <0 = skip structure info, structure not operating; 0 = rule-based structure; >0 = historical/other-data data array, flag=1 normally in this case, but is >1 for aggregated SFWMM structures;*/
 int histID;                    /*!< historical/other-data structure ID */
 int aggID;                     /*!< ID of SFWMM structure that holds aggregated data for current structure  */
 char S_nam[20];                /*!<  water control structure name */
 int str_cell_i;			/*!< row of the water control structure's cell  */
 int str_cell_j;			/*!< column of the water control structure's cell  */
 int canal_fr;		        /*!< number-ID of the donor canal */
 int canal_to;                  /*!< number-ID of the recipient canal */
 int cell_i_fr;		        /*!< row of the donor water control structure's cell */
 int cell_j_fr;		        /*!< column of the donor water control structure's cell */
 int cell_i_to;		        /*!< row of the recipient water control structure's cell */
 int cell_j_to;		        /*!< column of the recipient water control structure's cell */
 float HW_stage;		/*!< UNUSED static target stage in canal or cell upstream (+/-) (positive if stage should be > targeted and a minus otherwise) */
 float TW_stage;		/*!< UNUSED static target stage in canal or cell downstream (+/-) (positive if stage should be > targeted and a minus otherwise) */
 struct Schedule *HW_graph;	/*!< pointer to array of data if headwater target stage is to be a temporal graph */
 struct Schedule *TW_graph;	/*!< pointer to array of data if tailwater target stage is to be a temporal graph */
 int cell_i_HW;			/*!< UNUSED row of the headwater cell that may influence the operation */
 int cell_j_HW;			/*!< UNUSED column of the headwater cell that may influence the operation */
 int cell_i_TW;			/*!< UNUSED row of the tailwater cell that may influence the operation */
 int cell_j_TW;			/*!< UNUSED column of the tailwater cell that may influence the operation */
 float w_coef;			/*!< UNUSED weir coefficient (=0 for virtual structure) */
 double flow;			/*!< flow through structure (m^3/d) */
 int TPser;                     /*!< TP conc: flag to indicate use of concentration that is: -1=calc'd, 0= constant, 1=time series  */
 int TNser;                     /*!< UNUSED TN conc: flag to indicate use of concentration that is: -1=calc'd, 0= constant, 1=time series  */
 int TSser;                     /*!< TS (Total Salt) conc: flag to indicate use of concentration that is: -1=calc'd, 0= constant, 1=time series  */
 float TP;                      /*!< historical TP concentration, constant value across time */
 float TN;                      /*!< UNUSED historical TN concentration, constant value across time */
 float TS;                      /*!< historical TS (salt) concentration, constant value across time */
 double conc_P;			/*!< Phosphorus conc in flow through structure (g/L = kg/m3) */
 double conc_N;			/*!< Nitrogen conc in flow through structure (g/L = kg/m3) */
 double conc_S;			/*!< Salt conc in flow through structure (g/L = kg/m3) */
 double SumFlow;		/*!< sum of flow through structure over the summary interval (m^3) */
 double Sum_P;			/*!< TP mass sum (conc_P*flow) in flow through structure over the summary interval (kg) */
 double Sum_N;			/*!< UNUSED nitrogen mass sum (conc_N*flow) in flow through structure over the summary interval (kg) */
 double Sum_S;			/*!< salt mass sum (conc_S*flow) in flow through structure over the summary interval (kg) */
 int multiOut;     	        /*!< flag to indicate a structure has been processed as multiple outflow from one canal */
 struct Structure *next_in_list; /*!< points to the next structure in list */
};
struct Structure *struct_first;
struct Structure *structs;

/*! Water control structure schedule */
struct Schedule
{
 int num_point; 		 /*!< number of points in graph */
 struct Points *graph_points;
};
 
/*! Data points for water control structure schedule */
struct Points
{
 float time; 		 /*!< julian day (recurring years) */
 float value; 		 /*!< target stage height */
}; 

/*! Historical (or other model) water control structure flow data attributes */
struct HistData
{
     char S_nam[20];       /*!< water control structure name (also in the struct structure ) */
     float *arrayWat;		/*!< water flow data, m^3/day */
     float *arrayP;			/*!< phosphorus concentration for flow data, g/L=kg/m^3  */
     float *arrayN;			/*!< UNUSED nitrogen concentration for flow data, g/L=kg/m^3  */
     float *arrayS;			/*!< salt/tracer concentration for flow data, g/L=kg/m^3  */
     int aggID;            /*!< the SFWMM structure ID of the aggregated flow (e.g., S10's ID stored with S10A-D) */
     int aggCnt;           /*!< the # of dissaggregated structures contained in an aggregated SFWMM (e.g., S10 will have 3) */
     int flag;				/*!< the flag to indicate a structure is turned off (negative), rule-based (0), or data-driven (positive) */
};
struct HistData Hist_data[MAX_H_STRUCT]; /*!< pointer to array of pointers for the water control structure time series */
			   

void CanalReInit();
void Canal_Network_Init(float baseDatum, float *elev ); 
void Run_Canal_Network(float *SWH, float *ElevMap, float *MC, 
		       float *GWV, float *poros, float *GWcond, 
                       double *NA, double *PA, double *SA, 
		       double *GNA, double *GPA, double *GSA,
                       float *Unsat, float *sp_yield);  
void ReadStructures(char* name, float BASE_DATUM );
void Channel_configure(float *ElevMap, struct Chan *channel_first );
void getCanalElev (int chan_n);
void FluxChannel(int chan_n, float *SWH, float *ElevMap, float *MC, 
		 float *GWH, float *poros, float *GWcond, double *NA, 
		 double *PA, double *SA, double *GNA, double *GPA, 
		 double *GSA, float *Unsat, float *sp_yield);      
void Flows_in_Structures(float *SWH, float *Elev, float *MC, 
		         double *NA, double *PA, double *SA ); 

float f_Manning(float delta, float SWater, float SW_coef );
float f_Ground(float dh, float height, float GW_coef, float l_Length );
float GetGraph(struct Schedule *graph, float x );

struct Chan *ReadChanStruct(char* filename );
struct Cells *MarkCell(int x, int y, int index, float length, 
		       struct Cells *cell, int ch_number, int levee, 
		       int xm, int ym, int c_num, int *marked, float distTot );
struct Schedule *Read_schedule(char *sched_name, char *filename, float BASE_DATUM );

int Wrdcmp(char *s, char *t );
float UTM2kmx(double UTM );
float UTM2kmy(double UTM );


/* externs */

/* from driver_utilities.h */
extern char *ModelPath, *OutputPath, *ProjName;
extern char modelName[20], modelVers[10];

extern double julday(int mon, int day, int year, int h, int mi, double se);
extern void init_pvar(VOIDP Map, UCHAR* mask, unsigned char Mtype,float iv);
extern VOIDP nalloc(unsigned mem_size, const char var_name[]);
extern char *Scip(char *s, char SYM );
extern int isInteger(char *target_str);
extern int isFloat(char *target_str); 
extern float FMOD( float x, float y); 
extern int Round(float x);

/* from generic_driver.h */
extern char SimAlt[20], SimModif[20];        
extern double Jdate_init, Jdate_end;  
extern int PORnumday;
extern float can_Intvl;
extern int canPrint;

/* from unitmod.h */
extern unsigned char *HAB;
extern int *basn;
extern float GP_DetentZ, GP_MinCheck, GP_SLRise;
extern float  GP_mannDepthPow, GP_mannHeadPow, GP_calibGWat;

/* from fluxes.h */
extern float Flux_SWcells(int i0,int i1,int j0,int j1,
		          float *SWater,float *Elevation,float *MC);
extern void Flux_SWstuff(int i0,int i1,int j0,int j1, float Flux,
                         float *SURFACE_WAT, double *STUF1, double *STUF2, double *STUF3);

/* from budgstats.h */
extern int numBasn;
extern basnDef **basn_list;
extern basnDef *basins;

/* from budgstats_birvars.h */
extern double *TOT_VOL_CAN;         /* total canal volume, units = m^3 */

/* basin-based, horiz inputs to canal and cells, units = m^3 */
extern double *VOL_IN_STR, *VOL_IN_OVL, *VOL_IN_SPG, *VOL_IN_GW;
extern double *VOL_OUT_STR, *VOL_OUT_OVL, *VOL_OUT_SPG, *VOL_OUT_GW;

extern double *TOT_P_CAN;          /* total canal P, units = kg */

/* basin-based, horiz inputs to canal and cells, units = kg */
extern double *P_IN_STR, *P_IN_OVL, *P_IN_SPG, *P_IN_GW;
extern double *P_OUT_STR, *P_OUT_OVL, *P_OUT_SPG, *P_OUT_GW;

extern double *TOT_S_CAN;          /* total canal S, units = kg */

/* basin-based, horiz inputs to canal and cells, units = kg */
extern double *S_IN_STR, *S_IN_OVL, *S_IN_SPG, *S_IN_GW;
extern double *S_OUT_STR, *S_OUT_OVL, *S_OUT_SPG, *S_OUT_GW;

#endif /* WATMGM_H */
